/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/*
 * \ingroup lardis_operators
 * \defgroup shift_sar subpixel correlation method
 */

#ifndef LAR_TCHEBY_POLY_APPROX_H
#define LAR_TCHEBY_POLY_APPROX_H

#include <stdlib.h>
#include<stdio.h>
#include<math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_math.h"

struct st_rationel {
  int num;
  int den;
};
typedef struct st_rationel rat;

struct st_vecteur {
  int *ptv;
  int dim; 
};
typedef struct st_vecteur vect;

struct st_matrice {
  int coef;
  int dim;
  int **ptm;
};
typedef struct st_matrice mat;

struct st_matrice_fl {
  int coef;
  int dim;
  float **ptm;
};
typedef struct st_matrice_fl mat_fl;

/**
* \ingroup tcheby_poly_approx
* \typedef typedef Tcheby_poly_approx_param
*
* \brief structure that contains the parameters of the operator
*
*/
typedef struct struct_tcheby_poly_approx_param
{
  int NV; /*size of approximation window*/
  int K; /*degree of polynomial*/
}tcheby_poly_approx_param, *Tcheby_poly_approx_param;

/**
 * \ingroup tcheby_poly_approx
 * \fn Tcheby_poly_approx_param new_tcheby_poly_approx_param();
 *
 * \return A new reference on allocated tcheby_poly_approx_param structure
 *
 * \brief Create a new Tcheby_poly_approx_param reference
 *
 */

static  Tcheby_poly_approx_param new_tcheby_poly_approx_param();

/**
 * \ingroup tcheby_poly_approx
 * \fn void free_tcheby_poly_approx_param(Tcheby_poly_approx_param p);
 *
 * \param p A reference on a tcheby_poly_approx_param structure
 *
 * \brief Free an allocated Tcheby_poly_approx_param reference
 *
 */

void free_tcheby_poly_approx_param(Tcheby_poly_approx_param p);

/**
* \ingroup tcheby_poly_approx
* \fn void define_tcheby_poly_approx_param(char *extra_description);
* \brief Default efidir function that defines parameters
*
* \param extra_description Extra description for multi definition of the same parameter(NULL is accepted)
*
*/
void define_tcheby_poly_approx_param(char *extra_description);

/**
* \ingroup tcheby_poly_approx
* \fn Tcheby_poly_approx_param get_tcheby_poly_approx_param();
* \brief Default efidir function that gets parameters
*
* \return a Tcheby_poly_approx_param that contains the parameters
*/
Tcheby_poly_approx_param get_tcheby_poly_approx_param();

/**
* \ingroup tcheby_poly_approx
* \fn void tcheby_masks(Tcheby_poly_approx_param p,EFIDIRImage masksImage_num,EFIDIRImage masksImage_den);
* \brief surface approximation by Tchebychev polynomial
*
* \param p is the structure parameter for the operator
* \param masksImage is the image which contains masks for calculate polynomial coefficient
* \param masksImage_num is the image which contains masks denominator for calculate polynomial coefficient
* \param masksImage_den is the image which contains masks numerator for calculate polynomial coefficient
*
*/
void tcheby_masks(Tcheby_poly_approx_param p,EFIDIRImage masksImage_num,EFIDIRImage masksImage_den);


/**
* \ingroup tcheby_poly_approx
* \fn void tcheby_poly_approx(Tcheby_poly_approx_param p,EFIDIRImage inputImage,EFIDIRImage masksImage_num,EFIDIRImage masksImage_den);
* \brief surface approximation around the maximum value of image by Tchebychev polynomial
*
* \param p is the structure parameter for the operator
* \param inputImage is the image which it maximum has approached by polynomial approximation
* \param masksImage_num is the image which contains masks denominator for calculate polynomial coefficient
* \param masksImage_den is the image which contains masks numerator for calculate polynomial coefficient
* 
*/
void tcheby_poly_approx(Tcheby_poly_approx_param p,EFIDIRImage inputImage,EFIDIRImage masksImage_num,EFIDIRImage masksImage_den);

void r_mul(rat *ptr, rat *a, rat *b);
void r_add(rat *ptr, rat *a, rat *b);
void r_quot(rat *ptr, int a, int b);
void r_simplif(rat *ptr);
void p_val(rat *res,rat *poly,int deg, rat *x);
int pgcd(int a,int b);
int n_pgcd(int *pt,int n);
void r_fact(vect *v_res,rat *ptr);
static void pos_max_image(EFIDIRImage input, int *X, int *Y);

#endif/*COM_TCHEBY_POLY_APPROX_H*/
