/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup simul_camera_image simulate camera image using DEM
 */
#ifndef LAR_SIMUL_CAMERA_IMAGE_H
#define LAR_SIMUL_CAMERA_IMAGE_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include "efidir_image_list.h"

#include "lar_dem_camera_visu.h"


#include "../../include/sarlut/slt_calculations_geodetic.h"
#include "../../include/sarlut/slt_geom2D.h"
// #include "../../include/sarlut/slt_simu.h"

/**** dcalered in "lar_dem_camera_visu.h"
#define NOM_FICH_BIS 256
#define PIXEL_SIZE_DEM 10
#define RTerrestre (6366000.0)
#define WGSa (6378137.0)
#define WGSb (6356752.3142)
#define WGSf (1/298.257223563)

typedef struct{
 double east_x;
 double north_y;
 double size_x;
 double size_y;
 double rotation;
}TMapInfoLambert;

typedef struct{
 double X;
 double Y;
 double Z;
}TXYZ;

typedef struct{
 int x;
 int y;
 float z;
}discrete_coord_3d;
*/
/**
 * \ingroup simul_camera_image
 * \typedef Simul_camera_image_param
 *
 * \brief reference on simul_camera_image_param structure
 *
 */
/**
 * \ingroup simul_camera_image
 * \struct simul_camera_image_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct struct_simul_camera_image_param{
  int verbose; /*!< verbose*/
  double camera_north; /*!< north lambert II coordinate of camera*/
  double camera_east; /*!< east lambert II coordinate of camera*/
  double camera_up; /*!< up lambert II coordinate of camera*/
  float camera_inclination_angle; /*!< inclination angle of camera line of sight*/
  float camera_north_angle; /*!< camera line of sight angle in relation to north direction*/
  float camera_side_angle; /*!< side angle of camera */
  int camera_los_direction; /*!< direction of camera line of sight*/
  int camera_image_nb_col; /*!< columns number of camera image*/
  int camera_image_nb_lin; /*!< lines number of camera image*/
  float camera_ccd_sensor_l; /*!< width of camera CCD sensor (columns)*/
  float camera_ccd_sensor_h; /*!< height of camera CCD sensor (lines)*/
  float camera_focal; /*!< focal length of camera */
  int camera_image_ratio; /*!< camera image ratio */
  float col_offset_correction; /*!< column offset correction to match synthetic image with real image*/
  float lin_offset_correction; /*!< line offset correction to match synthetic image with real image*/
  float col_ratio_correction; /*!< column ratio correction to match synthetic image with real image*/
  float lin_ratio_correction; /*!< line ratio correction to match synthetic image with real image*/
  int cut_synthetic_image; /*!< cut synthetic image with image sizes*/
}simul_camera_image_param, *Simul_camera_image_param;

/**
 * \ingroup simul_camera_image
 * \fn new_simul_camera_image_param()
 *
 * \return A new reference on allocated simul_camera_image_param structure
 *
 * \brief Create a new Simul_camera_image_param reference 
 *
 */
static Simul_camera_image_param new_simul_camera_image_param();

/**
 * \ingroup simul_camera_image
 * \fn free_simul_camera_image_param(Simul_camera_image_param p)
 *
 * \param p A reference on a simul_camera_image_param structure
 *
 * \brief Free an allocated Simul_camera_image_param reference 
 *
 */
void free_simul_camera_image_param(Simul_camera_image_param p);

/**
 * \ingroup simul_camera_image
 * \fn void define_simul_camera_image_param(char *extra_description)
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_simul_camera_image_param(char *extra_description);

/**
 * \ingroup simul_camera_image
 * \fn Simul_camera_image_param get_simul_camera_image_param()
 * \brief Default efidir function that gets parameters
 *
 * \return A Simul_camera_image_param that contains the parameters.
 */
Simul_camera_image_param get_simul_camera_image_param();

/**
 * \ingroup simul_camera_image
 * \fn void simul_camera_image(Simul_camera_image_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask , char *ascii_name, char *mask_list_name, EFIDIRImage DEMrefinedmask);
 * \brief function simulate camera image using DEM
 * Function simulating camera image using DEM and the projective geometry
 *
 * \param p is the structure of the parameters for the operator
 * \param DEMImage is the input DEM image
 * \param DEMmask is the input DEM mask image
 * \param ascii_name is the ascii file name for (x,y) cordinates of the perspective projection
 * \param mask_list_name is ...
 * \param DEMrefinedmask is the output DEM refined mask image  
 */
void simul_camera_image(Simul_camera_image_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask , char *ascii_name, char *mask_list_name, EFIDIRImage DEMrefinedmask);

/*
 * \ingroup simul_camera_image
 * \fn coord_3d unit_vector_construct_angle();
 * \brief 
 * Function...
 *
 * \param phi is the first 3d vector 
 * \param theta is the second 3d vector
 *
 * \return the 3d unit vector 
 *
 */
coord_3d unit_vector_construct_angle(double phi, double theta);

/*
 * \ingroup simul_camera_image
 * \fn coord_3d unit_vector_construct_angle();
 * \brief 
 * Function...
 *
 * \param a is the first 3d vector 
 * \param b is the second 3d vector
 *
 * \return the 3d vector cross product result 
 *
 */
// coord_3d unit_vector_construct_point();

/*
 * \ingroup simul_camera_image
 * \fn coord_3d cross_product(coord_3d a, coord_3d b);
 * \brief function calculate cross product between a and b 3d vectors
 * Function calculating cross product between a and b 3d vectors
 *
 * \param a is the first 3d vector
 * \param b is the second 3d vector
 *
 * \return the 3d vector cross product result 
 *
 */
coord_3d cross_product(coord_3d a, coord_3d b);

/*
 * \ingroup simul_camera_image
 * \fn coord_3d scalar_product(coord_3d a, coord_3d b);
 * \brief function calculate cross product between a and b 3d vectors
 * Function calculating scalar product between a 3d vector and b 
 *
 * \param a is the 3d vector
 * \param b is the scalar value
 *
 * \return the 3d vector scalar product result 
 *
 */
coord_3d scalar_product(coord_3d a, double b);

void cross_product_display(coord_3d a, coord_3d b, coord_3d result);

void vector_display(coord_3d a);

void matrix_transformation_display(double transformation[][4]);

void camera_position_lambert_bis(Simul_camera_image_param p, EFIDIRImage DEMImage,double *col_position,double *lin_position,int *round_col_position,int *round_lin_position,TMapInfoLambert *map_info);

void construct_rotation_matrix(coord_3d view, coord_3d lin, coord_3d col,double rotation[][3]);

void construct_translation_matrix(coord_3d view, coord_3d lin, coord_3d col,double tx,double ty,double tz,double *translation);

void construct_transformation_matrix(double rotation[][3],double *translation,double focal,double tz,double transformation[][4]);

void perspective_projection(Simul_camera_image_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,double transformation[][4],char *ascii_name, TMapInfoLambert map_info,char *mask_list_name, EFIDIRImage DEMrefinedmask);

#endif /* LAR_SIMUL_CAMERA_IMAGE_H*/
