/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup shift_sar sub-pixel dispalcement by maximum of different similarity criterion
 */

#ifndef LAR_SHIFT_SAR_H
#define LAR_SHIFT_SAR_H

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_math.h"


#define N_33 9
#define N_55 25
#define M 6
#define NV_33 3
#define NV_55 5
#define SUBPIXEL_THRESHOLD 1.0

//N_xx: nombre de lignes de la matrice a (=nbre N*NV) (xx = {33,55})
//M: nombre de colones de la matrice b (=nbre de coef du polynome)
//NV_xx: taille de la fenêtre carrée de voisinnage pour calcauler le polynome qui "fite" les valeurs de similaritées (xx = {33,55})
//SUBPIXEL_THRESHOLD : seuil sur la valeur maximale du résidu subpixelique


/**
 * \ingroup shift_sar
 * \file lar_shift_sar.h
 * \author Renaud Fallourd (LISTIC)
 *
 * \brief operator of sub-pixel dispalcement by maximum of different similarity criterion
 *
 * file includes (stucture, prototype de fonction) of the operator shift_sar
 *
 *
 */

typedef struct{
  float A_33[N_33][M];
  float A_55[N_55][M];
  float B_33[N_33];
  float B_55[N_55];
  float X[M];
}matrix_poly;

typedef struct{
  Complex8 vector_dpl; //Re: dx displacement and Im: dy displacement
  Complex8 fw3dBm; //Largeur à -3dB (Re: en x et Im: en y )(full width at -3dB maximum)
  float performance; // Performance de la mesure de déplacement.
  float sim_value; //Valeur de la maximisation de la fonction de similarité
  unsigned char subpx_flag; //Indique si la mesure est subpixelique (1: pas subpixelique car en butée de fenêtre - 2: pas subpixelique car interpollation mauvaise - 4: subpixelique !!!)
  float ml_offset; //Valeur maximale pour les critéres de type autre que corrélation (offset à retirer)
  int without_mask; // without_mask=1 if computation without mask - without_mask=0 if computation with mask
}MeasureInfo;

/**
 * \ingroup shift_sar
 * \typedef typedef shift_sar_param
 * \struct shift_sar_param
 *
 * \brief structure that contains the parameters of the operator nom
 *
 */
typedef struct struct_shift_sar_param{
  int method; /*!< method used to determinate shift between 2 images */
  int subpx_choice; /*!< choose between supixelic measurement or pixelic measurement  */
  int window_nb_r; /*!< row size of window (azimut) */
  int window_nb_c; /*!< column size of window (range) */
  int window_search_nb_r; /*!< row size of search window */
  int window_search_nb_c; /*!< column size of search window */
}shift_sar_param, *Shift_sar_param;

/**
 * \ingroup shift_sar
 * \fn new_shift_sar_param()
 *
 * \return A new reference on allocated shift_sar_param structure
 *
 * \brief Create a new shift_sar_param reference 
 *
 */
static Shift_sar_param new_shift_sar_param();

/**
 * \ingroup shift_sar
 * \fn free_shift_sar_param(Shift_sar_param p)
 *
 * \param p A reference on a shift_sar_param structure
 *
 * \brief Free an allocated Shift_sar_param reference 
 *
 */
void free_shift_sar_param(Shift_sar_param p);
// void free_nom_param(Shift_sar_param p);

/**
 * \ingroup shift_sar
 * \fn void define_shift_sar_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/
void define_shift_sar_param(char *extra_description);

/**
 * \ingroup shift_sar
 * \fn Shift_sar_param get_shift_sar_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Shift_sar_param that contains the parameters.
 */
Shift_sar_param get_shift_sar_param();

/**
 * \ingroup shift_sar
 * \fn void shift_sar(Shift_sar_param p, EFIDIRImage inputImage_1, EFIDIRImage inputImage_2, EFIDIRImage mask, EFIDIRImage outputImage_1, EFIDIRImage outputImage_2, EFIDIRImage outputImage_3,EFIDIRImage outputImage_4,EFIDIRImage outputImage_5);
 * \brief Function that calculates the displacement between two SAR intensity images by different method
 * longue description
 *
 * \param p is the parameters structure of operator.
 * \param inputImage_1 is the master image
 * \param inputImage_2 is the slave image
 * \param mask is the mask (value=0 for fix area and value=1 for moved area)
 * \param outputImage_1 is the displacement image
 * \param outputImage_2 is the similarity peak image
 * \param outputImage_3 is the sub-pixel mask
 * \param outputImage_4 is the confidence image
 * \param outputImage_5 ???
 *
 */
void shift_sar(Shift_sar_param p, EFIDIRImage inputImage_1, EFIDIRImage inputImage_2, EFIDIRImage mask, EFIDIRImage outputImage_1, EFIDIRImage outputImage_2, EFIDIRImage outputImage_3,EFIDIRImage outputImage_4,EFIDIRImage outputImage_5);

/*
 * \ingroup shift_sar
 * \fn static void zncc(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage inputMask, EFIDIRImage similarity_window, MeasureInfo *q, int ibcl, int jbcl, matrix_poly ABXpoly)
 * \brief function that calculate the displacement for one pixel between two intensity sar images
 *
 * \param p is the parameters structure of operator
 * \param starter_flag flag accelerate algorithm
 * \param inputImage1 is the master image
 * \param inputImage2 is the slave image
 * \param similarity_window is the similarity window
 * \param ouput_vector is the vector value (EFIDIRImage complex4 1x1)
 * \param ibcl is the image line number
 * \param jbcl is the image column number
 *
 */
static void zncc(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage inputMask, EFIDIRImage similarity_window, MeasureInfo *q, int ibcl, int jbcl, matrix_poly ABXpoly);

/*
 * \ingroup shift_sar
 * \fn static void ncc(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage inputMask, EFIDIRImage similarity_window, MeasureInfo *q, int ibcl, int jbcl, matrix_poly ABXpoly)
 * \brief function that calculate the displacement for one pixel between two intensity sar images
 *
 * \param p is the parameters structure of operator
 * \param starter_flag flag accelerate algorithm
 * \param inputImage1 is the master image
 * \param inputImage2 is the slave image
 * \param similarity_window is the similarity window
 * \param ouput_vector is the vector value (EFIDIRImage complex4 1x1)
 * \param ibcl is the image line number
 * \param jbcl is the image column number
 *
 */
static void ncc(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage inputMask, EFIDIRImage similarity_window, MeasureInfo *q, int ibcl, int jbcl, matrix_poly ABXpoly);

/*
 * \ingroup shift_sar
 * \fn void uml(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage similarity_window, vector_similarity output_vector_sim, int ibcl, int jbcl){
 * \brief function that calculate the displacement for one pixel between two intensity sar images
 *
 * \param p is the parameters structure of operator
 * \param inputImage1 is the master image
 * \param inputImage2 is the slave image
 * \param similarity_window is the similarity window
 * \param ouput_vector is the vector value + similarity value
 * \param ibcl is the image line number
 * \param jbcl is the image column number
 *
 */
static void uml(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage inputMask, EFIDIRImage similarity_window, MeasureInfo *q, int ibcl, int jbcl, matrix_poly ABXpoly);
static void znuml(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage inputMask, EFIDIRImage similarity_window, MeasureInfo *q, int ibcl, int jbcl, matrix_poly ABXpoly);

/*
 * \ingroup shift_sar
 * \fn void ml_corr(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage similarity_window,  vector_similarity output_vector_sim, int ibcl, int jbcl);
 * \brief function that calculate the displacement for one pixel between two intensity sar images
 *
 * \param p is the parameters structure of operator
 * \param inputImage1 is the master image
 * \param inputImage2 is the slave image
 * \param similarity_window is the similarity window
 * \param ouput_vector is the vector value (EFIDIRImage complex4 1x1)
 * \param ibcl is the image line number
 * \param jbcl is the image column number
 *
*/
static void cml(Shift_sar_param p, EFIDIRImage inputImage1, EFIDIRImage inputImage2, EFIDIRImage inputMask, EFIDIRImage similarity_window, MeasureInfo *q, int ibcl, int jbcl, matrix_poly ABXpoly);

/*
 * \ingroup shift_sar
 * \fn void vector_value(Shift_sar_param p, EFIDIRImage window, vector_similarity shift_vector_sim);
 * \brief function that determinate the sift vector value by maximizing the similarity
 *
 * \param p is the parameters structure of operator
 * \param window is the similarity window
 * \param shift_vector is the shift vector (EFIDIRImage colplex4 1x1)
 *
 */
static void vector_value(Shift_sar_param p, EFIDIRImage window, MeasureInfo *q,matrix_poly ABXpoly);

/*
 * \ingroup shift_sar
 * \fn float window_average(EFIDIRImage inputImage, int x_window, int y_window, int ibcl, int jbcl);
 * \brief function that calculate the average window for an image
 *
 * \param inputImage is the processed image
 * \param x_window is window number of lines
 * \param y_window is window number of columns
 * \param ibcl is the image line number
 * \param jbcl is the image column number
 *
 * \return The mean on the window of processed image
 */
static float window_average(Shift_sar_param p,EFIDIRImage inputImage, EFIDIRImage inputMask, int x_window, int y_window, int ibcl, int jbcl);

static float window_std(Shift_sar_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, int x_window, int y_window, int ibcl, int jbcl);

static float window_max_val(EFIDIRImage inputImage, int x_window, int y_window, int ibcl, int jbcl);

static void resolveAXB_96(float A[9][6],float B[9],float X[6],int Alines,int Acolumns);

static void resolveAXB_256(float A[25][6],float B[25],float X[6],int Alines,int Acolumns);

static void inverse_66(float mat[M][M], int l, int c, float mat_inv[M][M]);

static void sub_vector_mix(Shift_sar_param p,EFIDIRImage window,Complex4 vec,matrix_poly ABXpoly,unsigned char *flag_sub,Complex8 *vec_sub,Complex8 *vec_fwhm);

static void sub_vector_33(Shift_sar_param p,EFIDIRImage window,Complex4 vec,matrix_poly ABXpoly,unsigned char *flag_sub,Complex8 *vec_sub,Complex8 *vec_fwhm);

static void sub_vector_55(Shift_sar_param p,EFIDIRImage window,Complex4 vec,matrix_poly ABXpoly,unsigned char *flag_sub,Complex8 *vec_sub,Complex8 *vec_fwhm);

static void fwhm(float X[6],float x_0,float y_0,Complex8 *vec_fwhm);

#endif /* LAR_SHIFT_SAR_H */
