/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup regul_markov displacement image regularization using Markov chain
 */
#ifndef COM_REGUL_MARKOV_H
#define COM_REGUL_MARKOV_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"


/**
 * \ingroup regul_markov
 * \typedef Regul_markov_param
 *
 * \brief reference on regul_markov_param structure
 *
 */
/**
 * \ingroup regul_markov
 * \struct regul_markov_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct struct_regul_markov_param{
  int index; /*!< index of */
  int window_search_nb_lines;
  int window_search_nb_columns;
}regul_markov_param, *Regul_markov_param;

/**
 * \ingroup regul_markov
 * \fn new_regul_markov_param()
 *
 * \return A new reference on allocated regul_markov_param structure
 *
 * \brief Create a new Regul_markov_param reference 
 *
 */
static Regul_markov_param new_regul_markov_param();

/**
 * \ingroup regul_markov
 * \fn free_regul_markov_param(Regul_markov_param p)
 *
 * \param p A reference on a regul_markov_param structure
 *
 * \brief Free an allocated Regul_markov_param reference 
 *
 */
void free_regul_markov_param(Regul_markov_param p);

/**
 * \ingroup regul_markov
 * \fn void define_regul_markov_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_regul_markov_param(char *extra_description);

/**
 * \ingroup regul_markov
 * \fn Regul_markov_param get_regul_markov_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Regul_markov_param that contains the parameters.
 */
Regul_markov_param get_regul_markov_param();

/**
 * \ingroup regul_markov
 * \fn void regul_markov(Regul_markov_param p, EFIDIRImage inputImg, EFIDIRImage inputMask, EFIDIRImage inputConf, EFIDIRImage outputImg);
 * \brief function of spatial regularization of displacement field by Markov chain
 * Function allowing to generate a spatial regularization image of 2D displacement image (complex float data type)
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImg is the complex float image to input (i.e. displacement vector)
 * \param inputMask is the mask of the input image
 * \param inputConf is the confidence image
 * \param outputImg is the regulated image correspond to input image
 *
 */
void regul_markov(Regul_markov_param p, EFIDIRImage inputImg, EFIDIRImage inputRandom, EFIDIRImage inputConf, EFIDIRImage outputImg);

/**
 * \ingroup regul_markov
 * \fn void init_random_image(EFIDIRImage inputImg, int lower_bound, int upper_bound);
 * \brief function generate random image field
 * Function allowing to generate a random image field
 *
 * \param inputImg is the int image to input
 * \param lower_bound is the lower bound value of random field
 * \param upper_bound is the upper bound value of random field
 *
 */
void init_random_image(EFIDIRImage inputImg, int lower_bound, int upper_bound);

/*
 * \ingroup regul_markov
 * \fn void random_image(EFIDIRImage inputImg, int lower_bound, int upper_bound);
 * \brief function generate random image field
 * Function allowing to generate a random image field
 *
 * \param inputImg is the int image to input
 * \param lower_bound is the lower bound value of random field
 * \param upper_bound is the upper bound value of random field
 *
 */
// void energy_clique(EFIDIRImage inputImg, int iBoucle, int jBoucle, int clique_choice){
// 
//   switch(clique_choice){
//     case '4':
//       inputImg[iBoucle-1][jBoucle-1] = 0;
//       break;
//     case '8':
//       inputImg[iBoucle-1][jBoucle-1] = 0;
//       break;
//     default:
//       printf("Clique choice is wrong\n");
//   }
// 
// }

#endif
