/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup recal_image image registration using normalized cross correlation
 */
#ifndef LAR_RECAL_IMAGE_H
#define LAR_RECAL_IMAGE_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include "com_crop.h"

typedef struct{
  int dx; // column
  int dy; // line
}TRecalVector;

typedef struct{
  float dx; // column
  float dy; // line
}TRecalVectorfl;

typedef struct{
  int first_col;
  int first_lig;
  int last_col;
  int last_lig;
}TCropInfo;

/**
 * \ingroup recal_image
 * \typedef Recal_image_param
 *
 * \brief reference on recal_image_param structure
 *
 */
/**
 * \ingroup recal_image
 * \struct recal_image_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct struct_recal_image_param{
  int verbose; /*!< Verbose option for debug*/
  int first_line; /*!< first line (left top line)*/
  int first_column; /*!< first column (left top column) */
  int last_line; /*!< last line (right top line) */
  int last_column; /*!< last column (right top column) */
  int window_search_lines; /*!< number of lines for window search */
  int window_search_columns; /*!< number of columns for window search */
}recal_image_param, *Recal_image_param;

/**
 * \ingroup recal_image
 * \fn new_recal_image_param()
 *
 * \return A new reference on allocated recal_image_param structure
 *
 * \brief Create a new Recal_image_param reference 
 *
 */
static Recal_image_param new_recal_image_param();

/**
 * \ingroup recal_image
 * \fn free_recal_image_param(Recal_image_param p)
 *
 * \param p A reference on a recal_image_param structure
 *
 * \brief Free an allocated Recal_image_param reference 
 *
 */
void free_recal_image_param(Recal_image_param p);

/**
 * \ingroup recal_image
 * \fn void define_recal_image_param(char *extra_description)
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_recal_image_param(char *extra_description);

/**
 * \ingroup recal_image
 * \fn Recal_image_param get_recal_image_param()
 * \brief Default efidir function that gets parameters
 *
 * \return A Recal_image_param that contains the parameters.
 */
Recal_image_param get_recal_image_param();

/**
 * \ingroup recal_image
 * \fn void recal_image(Recal_image_param p, EFIDIRImage MasterImg, EFIDIRImage SlaveImg, TRecalVector *vector)
 * \brief function of image co-registration
 * Function allowing to image co-registration
 *
 * \param p is the structure of the parameters for the operator
 * \param MasterImg is the master image input
 * \param SlaveImg is the slave image input
 * \param vector co-registration vector
 *
 */
void recal_image(Recal_image_param p, EFIDIRImage MasterImg, EFIDIRImage SlaveImg, TRecalVector *vector);

void crop_size(TRecalVector vector, TCropInfo *master, TCropInfo *slave, int master_nb_line, int master_nb_column, int slave_nb_line, int slave_nb_column);

void crop_size_list(int verbose,TRecalVector *memo_vector, int nb_list_image,int slave_nb_line,int slave_nb_column, TCropInfo *crop_list_slave);

void crop_image_recal(TCropInfo image_crop_info, EFIDIRImage InputImage, EFIDIRImage OutputImage);

#endif /* LAR_RECAL_IMAGE_H */
