/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup feasible_dpl_measurement_map feasible dpl measurement map : image which mapped where the displacement measurement is possible
 */
#ifndef LAR_FEASIBLE_DPL_MEASUREMENT_MAP_H
#define LAR_FEASIBLE_DPL_MEASUREMENT_MAP_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include "efidir_image_list.h"
#include "lar_recal_image.h"

/**
 * \ingroup feasible_dpl_measurement_map
 * \typedef Feasible_dpl_measurement_map_param
 *
 * \brief reference on feasible_dpl_measurement_map_param structure
 *
 */
/**
 * \ingroup feasible_dpl_measurement_map
 * \struct feasible_dpl_measurement_map_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct struct_feasible_dpl_measurement_map_param{
  int first_line; /*!< first line (left top line)*/
  int first_column; /*!< first column (left top column) */
  int nb_line; /*!< number of lines to keep */
  int nb_column; /*!< number of columns to keep */
  int first_line_bis; /*!< first line bis (left top line)*/
  int first_column_bis; /*!< first column bis (left top column) */
  int nb_line_bis; /*!< number of lines bis to keep */
  int nb_column_bis; /*!< number of columns bis to keep */
  float threshold; /*!< threshold for selected feasible displacement measurement */
  int verbose; /*!< Verbose option for debug*/ 
}feasible_dpl_measurement_map_param, *Feasible_dpl_measurement_map_param;

/**
 * \ingroup feasible_dpl_measurement_map
 * \fn new_feasible_dpl_measurement_map_param()
 *
 * \return A new reference on allocated feasible_dpl_measurement_map_param structure
 *
 * \brief Create a new Feasible_dpl_measurement_map_param reference 
 *
 */
static Feasible_dpl_measurement_map_param new_feasible_dpl_measurement_map_param();

/**
 * \ingroup feasible_dpl_measurement_map
 * \fn free_feasible_dpl_measurement_map_param(Feasible_dpl_measurement_map_param p)
 *
 * \param p A reference on a feasible_dpl_measurement_map_param structure
 *
 * \brief Free an allocated Feasible_dpl_measurement_map_param reference 
 *
 */
void free_feasible_dpl_measurement_map_param(Feasible_dpl_measurement_map_param p);

/**
 * \ingroup feasible_dpl_measurement_map
 * \fn void define_feasible_dpl_measurement_map_param(char *extra_description)
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_feasible_dpl_measurement_map_param(char *extra_description);

/**
 * \ingroup feasible_dpl_measurement_map
 * \fn Feasible_dpl_measurement_map_param get_feasible_dpl_measurement_map_param()
 * \brief Default efidir function that gets parameters
 *
 * \return A Feasible_dpl_measurement_map_param that contains the parameters.
 */
Feasible_dpl_measurement_map_param get_feasible_dpl_measurement_map_param();

/**
 * \ingroup feasible_dpl_measurement_map
 * \fn void feasible_dpl_measurement_map(Feasible_dpl_measurement_map_param p,EFIDIRListImage peakImages,EFIDIRListImage vecImages,EFIDIRImage inputMask,EFIDIRImage inputDistance,EFIDIRImage inputEast,EFIDIRImage inputNorth,EFIDIRImage inputAltitude,char *input_ascii,EFIDIRImage outputImage,EFIDIRImage outputVecImage,char *output_ascii,char *output_ascii_bis,char *output_correc_dpl);
 * \brief function of image co-registration
 * Function allowing to mapped where the displacement measurement is possible
 *
 * \param p is the structure of the parameters for the operator
 * \param peakImages is the list peak images
 * \param vecImages is the list displacement vector images
 * \param inputMask is ...
 * \param inputDistance is ...
 * \param inputEast is ...
 * \param inputNorth is ...
 * \param inputAltitude is ...
 * \param input_ascii is ...
 * \param outputImage is the output images
 * \param outputVecImage is ...
 * \param output_ascii is ...
 * \param output_ascii_bis is ...
 * \param output_correc_dpl is ...
 *
 */
void feasible_dpl_measurement_map(Feasible_dpl_measurement_map_param p,EFIDIRListImage peakImages,EFIDIRListImage vecImages,EFIDIRImage inputMask,EFIDIRImage inputDistance,EFIDIRImage inputEast,EFIDIRImage inputNorth,EFIDIRImage inputAltitude,char *input_ascii,EFIDIRImage outputImage,EFIDIRImage outputVecImage,char *output_ascii,char *output_ascii_bis,char *output_correc_dpl);

void crop_feasible_image_recal(TCropInfo image_crop_info, EFIDIRImage inputImage, EFIDIRImage outputImage);

#endif /* LAR_FEASIBLE_DPL_MEASUREMENT_MAP_H */
