/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup extract_dem create a DEM with a mosaïc
 */
#ifndef LAR_EXTRACT_DEM_H
#define LAR_EXTRACT_DEM_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include "efidir_image_list.h"

#define SIZE_STRING 256
#define IGNORED_VAL 9000

typedef struct{
 double east_x;
 double north_y;
 double size_x;
 double size_y;
 double rotation;
}TMapInfoLambert;

typedef struct{
 int nb_columns;
 int nb_lines;
 int x_start;
 int y_start;
 int step;
 int no_data;
}THeaderASCII;

/**
 * \ingroup extract_dem
 * \typedef Extract_dem_param
 *
 * \brief reference on extract_dem_param structure
 *
 */
/**
 * \ingroup extract_dem
 * \struct extract_dem_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct struct_extract_dem_param{
  int verbose; /*!< verbose */
  int ignored_val; /*!< value in the DEM that should be ignored */
}extract_dem_param, *Extract_dem_param;

/**
 * \ingroup extract_dem
 * \fn new_extract_dem_param()
 *
 * \return A new reference on allocated extract_dem_param structure
 *
 * \brief Create a new Extract_dem_param reference 
 *
 */
static Extract_dem_param new_extract_dem_param();

/**
 * \ingroup extract_dem
 * \fn free_extract_dem_param(Extract_dem_param p)
 *
 * \param p A reference on a extract_dem_param structure
 *
 * \brief Free an allocated Extract_dem_param reference 
 *
 */
void free_extract_dem_param(Extract_dem_param p);

/**
 * \ingroup extract_dem
 * \fn void define_extract_dem_param(char *extra_description)
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_extract_dem_param(char *extra_description);

/**
 * \ingroup extract_dem
 * \fn Extract_dem_param get_extract_dem_param()
 * \brief Default efidir function that gets parameters
 *
 * \return A Extract_dem_param that contains the parameters.
 */
Extract_dem_param get_extract_dem_param();

/**
 * \ingroup dem_camera_image
 * \fn void extract_dem(Extract_dem_param p,char *list_ascii_name,char *output_img_name);
 * \brief function determinate area cover by cameara in DEM 
 * Function allowing to determinate area cover by camera in DEM
 *
 * \param p is the structure of the parameters for the operator
 * \param list_ascii_name is the ASCII file name for (x,y) cordinates of the perspective projection
 * \param output_img_name is the output image name
 * void write_georef(EFIDIRImage output_img,THeaderASCII shape_dem)
 */
void extract_dem(Extract_dem_param p,char *list_ascii_name,char *output_img_name);

char **read_ascii_list(char *list_ascii_name,int *ascii_files_number);

void extract_ascii_headers(char *asci_name_table,THeaderASCII *ascii_hdrs);

void find_dem_size(int ascii_files_number,THeaderASCII *ascii_hdrs_table,THeaderASCII *shape_dem);

void extract_ascii_data(Extract_dem_param p,char **ascii_name_table,int ascii_files_number,THeaderASCII shape_dem,EFIDIRImage output_img);

void write_georef(char *output_img_name,THeaderASCII shape_dem);

#endif /* LAR_EXTRACT_DEM_H */
