/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup dist_corr subpixel correlation
 */

#ifndef LAR_DIST_CORR_H
#define LAR_DIST_CORR_H

#include <stdlib.h> 
#include <stdarg.h>

#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_math.h"
#include "efidir_system.h"
#include "efidir_string.h"
#include "efidir_matrix.h"

/**
 * \ingroup dist_corr
 * \file lar_dist_corr.h
 * \author Flavien Vernier (LISTIC)
 *
 * \brief operator of subpixel correlation
 *
 * file includes (stucture, prototype de fonction) of the operator dist_corr
 *
 *
 */

typedef struct{
  float module_vector;
  float orientation_vector;
}*displ_orien;


/**
 * \ingroup dist_corr
 * \enum SubPixelAlgorithm
 *
 * \brief Sub pixel computation options
 *
 */
enum SubPixelAlgorithm {
  NO_SUBPIXEL,        /*!< Do not compute sub pixel correlation */
  PARABOLIC3D,        /*!< Use a 3D parabol to compute sub pixel correlation */
  PARABOLIC2D,        /*!< Use a 2D parabol on x axe and another one on y axe to compute sub pixel correlation */
  BOTH3D2DPARABOLIC   /*!< Use a 3D parabol if possible or 2D parabols otherwise to compute sub pixel correlation */
};

/**
 * \ingroup dist_corr
 * \typedef typedef dist_corr_param
 *
 * \brief structure that contains the parameters of the operator dist_corr
 *
 */
typedef struct Dist_corr_param{
  int window_nb_r; /*!< row size of window (azimut) */
  int window_nb_c; /*!< column size of window (range) */
  int window_search_nb_r; /*!< row size of search window */
  int window_search_nb_c; /*!< column size of search window */
  int sub_pixel; /*!< sub-pixel correlation (see subpixel options)  */
  int step; /*!< step between computation points*/
  int process_number;
}dist_corr_param, *Dist_corr_param;

/**
 * \ingroup dist_corr
 * \fn Dist_corr_param new_dist_corr_param();
 *
 * \return A new reference on allocated dist_corr_param structure
 *
 * \brief Create a new Dist_corr_param reference
 *
 */

static Dist_corr_param new_dist_corr_param();

/**
 * \ingroup dist_corr
 * \fn void free_dist_corr_param(Dist_corr_param p);
 *
 * \param p A reference on a dist_corr_param structure
 *
 * \brief Free an allocated Dist_corr_param reference
 *
 */

void free_dist_corr_param(Dist_corr_param p);

/**
 * \ingroup dist_corr
 * \fn void define_dist_corr_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/
void define_dist_corr_param(char *extra_description);

/**
 * \ingroup dist_corr
 * \fn Dist_corr_param get_dist_corr_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Dist_corr_param that contains the parameters.
 */
Dist_corr_param get_dist_corr_param();

/**
 * \ingroup dist_corr
 * \fn void dist_corr(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by cross correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 *
 */
void dist_corr(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);




/**
 * \ingroup dist_corr
 * \fn void dist_corr_opt1(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by cross correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 *
 */
void dist_corr_opt1(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);



/**
 * \ingroup dist_corr
 * \fn void dist_corr_opt2(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by cross correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 *
 */
void dist_corr_opt2(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);


/**
 * \ingroup dist_corr
 * \fn void paradist_corr_opt1(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by cross correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 *
 */
void paradist_corr_opt1(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);

/**
 * \ingroup dist_corr
 * \fn void paradist_corr_opt3(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by cross correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 *
 */
void paradist_corr_opt3(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);

/**
 * \ingroup dist_corr
 * \fn void dist_corr_opt3(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by cross correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 *
 */
void dist_corr_opt3(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);

void dist_corr_hpeak_opt3(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM, EFIDIRImage correlationSubPixColorMapImage, EFIDIRImage correlationHPeakImage);

void sub_pixel_correlation(int subPixelOption,
			   double **correlationWindow, 
			   int maximumQ, int offsetQ, 
			   int maximumP, int offsetP,
			   int rPoint, int cPoint, 
			   EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM
			   );

/**
 * \ingroup dist_corr
 * \fn void dist_ZNCC(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by Zero-mean Normalized Cross Correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 * \param correlationSNR is the float output image of signal to noise ratio (value of the peak divided by the average of similarity values over the search window without considering the peak itself)
 */
void dist_ZNCC(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM, EFIDIRImage correlationSNR);

/**
 * \ingroup dist_corr
 * \fn void dist_ZNCC_opt1(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by Zero-mean Normalized Cross Correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 *
 */
void dist_ZNCC_opt1(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);

/**
 * \ingroup dist_corr
 * \fn void dist_ZNCC_opt2(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);
 * \brief function that calculate the dist by correlation between two parts of SAR images.
 *
 * This function calculates the dist by Zero-mean Normalized Cross Correlation between two parts of SAR images of amplitude.
 * The correlated parts is define only in master image by first and last line parameters.
 * The SAR images of amplitude may be computes in float and the output image
 * computes in complex float (cx8). The real part represents the dX
 * displacement and the imaginary part represents the dY displacement (in px).
 *
 * \param p structure that contains the parameters of the operator correlation
 * \param masterImage is the master image
 * \param slaveImage is the slave image
 * \param mask hide part of master image (can be null)
 * \param correlationDisplacement is the complex float output image of displacement
 * \param correlationPic is the float output image (pic of correlation)
 * \param correlationFWHM is the complex float output image of full width at half maximum
 *
 */
void dist_ZNCC_opt2(Dist_corr_param p, EFIDIRImage masterImage, EFIDIRImage slaveImage, EFIDIRImage mask, EFIDIRImage correlationDisplacement, EFIDIRImage correlationPic, EFIDIRImage correlationFWHM);

#endif
