/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup dem_camera_visu determinate area cover by camera in DEM
 */
#ifndef LAR_DEM_CAMERA_VISU_H
#define LAR_DEM_CAMERA_VISU_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"

#include "../../include/sarlut/slt_calculations_geodetic.h"
#include "../../include/sarlut/slt_geom2D.h"
// #include "../../include/sarlut/slt_simu.h"

#define NOM_FICH_BIS 256
#define PIXEL_SIZE_DEM 10
#define RTerrestre (6366000.0)
#define WGSa (6378137.0)
#define WGSb (6356752.3142)
#define WGSf (1/298.257223563)

typedef struct{
 double east_x;
 double north_y;
 double size_x;
 double size_y;
 double rotation;
}TMapInfoLambert;

typedef struct{
 double X;
 double Y;
 double Z;
}TXYZ;

typedef struct{
 int x;
 int y;
 float z;
}discrete_coord_3d;

/**
 * \ingroup dem_camera_visu
 * \typedef Dem_camera_visu_param
 *
 * \brief reference on dem_camera_visu_param structure
 *
 */
/**
 * \ingroup dem_camera_visu
 * \struct dem_camera_visu_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct struct_dem_camera_visu_param{
  int verbose; /*!< verbose*/
  double camera_north; /*!< north lambert II coordinate of camera*/
  double camera_east; /*!< east lambert II coordinate of camera*/
  double camera_up; /*!< up lambert II coordinate of camera*/
  float camera_inclination_angle; /*!< inclination angle of camera line of sight*/
  float camera_north_angle; /*!< camera line of sight angle in relation to north direction*/
  int camera_los_direction; /*!< direction of camera line of sight*/
  float camera_ccd_sensor_l; /*!< width of camera CCD sensor (columns)*/
  float camera_ccd_sensor_h; /*!< height of camera CCD sensor (lines)*/
  float camera_focal; /*!< focal length of camera*/
  int camera_image_ratio; /*!< camera image ratio*/
  float extra_horizontal_fov; /*!< pourcentage of extra horizontal field of view*/
}dem_camera_visu_param, *Dem_camera_visu_param;

/**
 * \ingroup dem_camera_visu
 * \fn new_dem_camera_visu_param()
 *
 * \return A new reference on allocated dem_camera_visu_param structure
 *
 * \brief Create a new Dem_camera_visu_param reference 
 *
 */
static Dem_camera_visu_param new_dem_camera_visu_param();

/**
 * \ingroup dem_camera_visu
 * \fn free_dem_camera_visu_param(Dem_camera_visu_param p)
 *
 * \param p A reference on a dem_camera_visu_param structure
 *
 * \brief Free an allocated Dem_camera_visu_param reference 
 *
 */
void free_dem_camera_visu_param(Dem_camera_visu_param p);

/**
 * \ingroup dem_camera_visu
 * \fn void define_dem_camera_visu_param(char *extra_description)
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_dem_camera_visu_param(char *extra_description);

/**
 * \ingroup dem_camera_visu
 * \fn Dem_camera_visu_param get_dem_camera_visu_param()
 * \brief Default efidir function that gets parameters
 *
 * \return A Dem_camera_visu_param that contains the parameters.
 */
Dem_camera_visu_param get_dem_camera_visu_param();

/**
 * \ingroup dem_camera_image
 * \fn void dem_camera_visu(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance);
 * \brief function determinate area cover by cameara in DEM 
 * Function allowing to determinate area cover by camera in DEM
 *
 * \param p is the structure of the parameters for the operator
 * \param DEMImage is the DEM image input
 * \param DEMmask is the DEM mask image output
 * \param DEMdistance is the DEM distance map image output
 *
 */
void dem_camera_visu(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance);

int read_georef(char *nom_fichier_georef, TMapInfo *map_info);

int read_georef_lambert(char *nom_fichier_georef, TMapInfoLambert *map_info);

double dist_orthodromique_metre(float rayon, TLatLong A, TLatLong B);

void lldeg_2_llrad(TLatLong *ll);

double rayon_terre(TLatLong A, float a, float b);

coord_3d llrad_2_llcart(TLatLong ll, float h, float a, float b);

/*
 * \ingroup dem_camera_image
 * \fn void camera_position(Dem_camera_visu_param p, EFIDIRImage DEMImage,int *col_position, int *lin_position);
 * \brief function determinate the position of the camera in DEM image
 * Function allowing to determinate area cover by camera in DEM
 *
 * \param p is the structure of the parameters for the operator
 * \param DEMImage is the DEM image input
 * \param col_position is the column position of the camera
 * \param lin_position is the line position of the camera
 *
 */
void camera_position_ll(Dem_camera_visu_param p, EFIDIRImage DEMImage,int *col_position, int *lin_position);
void camera_position_lambert(Dem_camera_visu_param p, EFIDIRImage DEMImage,float *col_position, float *lin_position,int *round_col_position, int *round_lin_position,TMapInfoLambert *map_info);

/*
 * \ingroup dem_camera_image
 * \fn void field_of_view_angle(Dem_camera_visu_param p, float *vertical_angle, float *horizontal_angle);
 * \brief function determinate the camera field of view angles 
 * Function determining the camera field of view angles according to the focal length and the image size (inpixel)
 *
 * \param p is the structure of the parameters for the operator
 * \param vertical_angle is the vertical angle of the field of view
 * \param horizontal_angle is the horizontal angle of the field of view
 *
 */
void field_of_view_angle(Dem_camera_visu_param p, float *vertical_angle, float *horizontal_angle);

/*
 * \ingroup dem_camera_image
 * \fn void camera_visu_sw(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance,float vertical_angle, float horizontal_angle, int pos_cam_lin, int pos_cam_col);
 * \brief function determinate the DEM pixels which are visible in camera image in North-East direction
 * Function determining the DEM pixels which are visible in camera image in North-East direction
 *
 * \param p is the structure of the parameters for the operator
 * \param DEMImage is the DEM image input
 * \param DEMmask is the mask output of the camera field of view in DEM geometry
 * \param DEMdistance is the distance image between camera and the DEM pixel according to DEMmask
 * \param vertical_angle is the vertical angle of the field of view
 * \param horizontal_angle is the horizontal angle of the field of view
 * \param lin_position is the line position of the camera
 * \param col_position is the column position of the camera
 *
 */
void camera_visu_ne(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance,float vertical_angle, float horizontal_angle, int pos_cam_lin, int pos_cam_col);

/*
 * \ingroup dem_camera_image
 * \fn void camera_visu_sw(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance,float vertical_angle, float horizontal_angle, int pos_cam_lin, int pos_cam_col);
 * \brief function determinate the DEM pixels which are visible in camera image in South-East direction
 * Function determining the DEM pixels which are visible in camera image in South-East direction
 *
 * \param p is the structure of the parameters for the operator
 * \param DEMImage is the DEM image input
 * \param DEMmask is the mask output of the camera field of view in DEM geometry
 * \param DEMdistance is the distance image between camera and the DEM pixel according to DEMmask
 * \param vertical_angle is the vertical angle of the field of view
 * \param horizontal_angle is the horizontal angle of the field of view
 * \param lin_position is the line position of the camera
 * \param col_position is the column position of the camera
 *
 */
void camera_visu_se(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance,float vertical_angle, float horizontal_angle, int pos_cam_lin, int pos_cam_col);

/*
 * \ingroup dem_camera_image
 * \fn void camera_visu_sw(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance,float vertical_angle, float horizontal_angle, int pos_cam_lin, int pos_cam_col);
 * \brief function determinate the DEM pixels which are visible in camera image in South-West direction
 * Function determining the DEM pixels which are visible in camera image in South-West direction
 *
 * \param p is the structure of the parameters for the operator
 * \param DEMImage is the DEM image input
 * \param DEMmask is the mask output of the camera field of view in DEM geometry
 * \param DEMdistance is the distance image between camera and the DEM pixel according to DEMmask
 * \param vertical_angle is the vertical angle of the field of view
 * \param horizontal_angle is the horizontal angle of the field of view
 * \param lin_position is the line position of the camera
 * \param col_position is the column position of the camera
 *
 */
void camera_visu_sw(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance,float vertical_angle, float horizontal_angle, float pos_cam_lin, float pos_cam_col, int round_pos_cam_lin, int round_pos_cam_col,TMapInfoLambert map_info);

/*
 * \ingroup dem_camera_image
 * \fn void camera_visu_sw(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance,float vertical_angle, float horizontal_angle, int pos_cam_lin, int pos_cam_col);
 * \brief function determinate the DEM pixels which are visible in camera image in North-West direction
 * Function determining the DEM pixels which are visible in camera image in North-West direction
 *
 * \param p is the structure of the parameters for the operator
 * \param DEMImage is the DEM image input
 * \param DEMmask is the mask output of the camera field of view in DEM geometry
 * \param DEMdistance is the distance image between camera and the DEM pixel according to DEMmask
 * \param vertical_angle is the vertical angle of the field of view
 * \param horizontal_angle is the horizontal angle of the field of view
 * \param lin_position is the line position of the camera
 * \param col_position is the column position of the camera
 *
 */
void camera_visu_nw(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask,EFIDIRImage DEMdistance,float vertical_angle, float horizontal_angle, int pos_cam_lin, int pos_cam_col);

/*
 * \ingroup dem_camera_image
 * \fn discrete_coord_3d* init_tab_segment(int xi,int yi, int xf, int yf,int *n_tab);
 * \brief function create (x,y,z) coordinates structure table
 * Function creating (x,y,z) coordinates structure table
 *
 * \param xi is the initial column position (i.e. camera)
 * \param yi is the initial line position (i.e. camera)
 * \param xf is the final column position (i.e. DEM pixel)
 * \param yf is the final line position (i.e. DEM pixel)
 * \param n_tab is the table size
 *
 * \return A discrete_coord_3d that contains the segment coordinates
 *
 */
discrete_coord_3d* init_tab_segment(int xi,int yi, int xf, int yf,int *n_tab);

/*
 * \ingroup dem_camera_image
 * \fn void segment_voxel(int xi,int yi, float zi,int xf,int yf,float zf,discrete_coord_3d *segment);
 * \brief function determinate (x,y,z) segment coordinate 
 * Function determinating (x,y,z) segment coordinates between camera and DEM pixel
 *
 * \param xi is the initial column position (i.e. camera)
 * \param yi is the initial line position (i.e. camera)
 * \param zi is the initial altitude (i.e. camera altitude)
 * \param xf is the final column position (i.e. DEM pixel)
 * \param yf is the final line position (i.e. DEM pixel)
 * \param zf is the final altitude (i.e. DEM pixel altitude)
 * \param segment is (x,y,z) coordinates of segment
 *
 */
void segment_voxel(int xi,int yi, float zi,int xf,int yf,float zf,discrete_coord_3d *segment);

/*
 * \ingroup dem_camera_image
 * \fn void distance_camera_pixel(Dem_camera_visu_param p, EFIDIRImage DEMmask, EFIDIRImage DEMdistance, int pos_cam_lin, int pos_cam_col);
 * \brief function determinate
 * Function determinating (x,y,z) segment coordinates between camera and DEM pixel
 *
 * \param p is the structure of the parameters for the operator
 * \param DEMmask is the mask output of the camera field of view in DEM geometry
 * \param DEMdistance is camera-pixel DEM distance image 
 *
 */
void distance_camera_pixel(Dem_camera_visu_param p, EFIDIRImage DEMImage, EFIDIRImage DEMmask, EFIDIRImage DEMdistance, float pos_cam_lin, float pos_cam_col, TMapInfoLambert map_info);

#endif /* LAR_DEM_CAMERA_VISU_H */
