/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup lardis_operators
 * \defgroup delux_median_filter delux median filter : image which mapped where the displacement measurement is possible
 */
#ifndef LAR_DELUX_MEDIAN_FILTER_H
#define LAR_DELUX_MEDIAN_FILTER_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include "efidir_image_list.h"

/**
 * \ingroup delux_median_filter
 * \typedef Delux_median_filter_param
 *
 * \brief reference on delux_median_filter_param structure
 *
 */
/**
 * \ingroup delux_median_filter
 * \struct delux_median_filter_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */

typedef struct struct_delux_median_filter_param{
  int window_nb_lines; /*!< lines number of the median filter window */
  int window_nb_columns; /*!< columns number of the median filter window */
  int median_choice; /*!< choice of median method */
  int verbose; /*!< Verbose option for debug*/
}delux_median_filter_param, *Delux_median_filter_param;

/**
 * \ingroup delux_median_filter
 * \fn new_delux_median_filter_param()
 *
 * \return A new reference on allocated delux_median_filter_param structure
 *
 * \brief Create a new Delux_median_filter_param reference 
 *
 */
static Delux_median_filter_param new_delux_median_filter_param();

/**
 * \ingroup delux_median_filter
 * \fn free_delux_median_filter_param(Delux_median_filter_param p)
 *
 * \param p A reference on a delux_median_filter_param structure
 *
 * \brief Free an allocated Delux_median_filter_param reference 
 *
 */
void free_delux_median_filter_param(Delux_median_filter_param p);

/**
 * \ingroup delux_median_filter
 * \fn void define_delux_median_filter_param(char *extra_description)
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_delux_median_filter_param(char *extra_description);

/**
 * \ingroup delux_median_filter
 * \fn Delux_median_filter_param get_delux_median_filter_param()
 * \brief Default efidir function that gets parameters
 *
 * \return A Delux_median_filter_param that contains the parameters.
 */
Delux_median_filter_param get_delux_median_filter_param();

/**
 * \ingroup delux_median_filter
 * \fn void delux_median_filter(Delux_median_filter_param p,EFIDIRImage inputImage,EFIDIRImage maskImage,EFIDIRImage outputImage);
 * \brief function of image co-registration
 * Function allowing to mapped where the displacement measurement is possible
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImage is the image input
 * \param maskImage is the mask input image
 * \param outputImage is the filtered output image
 *
 */
void delux_median_filter(Delux_median_filter_param p,EFIDIRImage inputImage,EFIDIRImage maskImage,EFIDIRImage outputImage);

void median_filter_float(Delux_median_filter_param p,int ibcl,int jbcl,EFIDIRImage inputImage,EFIDIRImage outputImage);

void median_filter_complex8(Delux_median_filter_param p,int ibcl,int jbcl,EFIDIRImage inputImage,EFIDIRImage outputImage);

#endif /* LAR_DELUX_MEDIAN_FILTER_H */
