/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */

/**
* \ingroup lardis_operators
* \defgroup compare_ground_sar ground measurements operators
*/

/**
* \ingroup compare_ground_sar
* \file lar_compare_ground_sar.h
* \author renaud fallourd
*
* \brief comparison of SAR displacment measurement with ground displacment measurement
* file includes structures, prototypes of function of the operator compare_ground_sar
*
*/

#ifndef LAR_COMPARE_GROUND_SAR_H
#define LAR_COMPARE_GROUND_SAR_H

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_math.h"
#include "efidir_assert.h"

#include "../../include/sarlut/slt_calculations_geodetic.h" //accés au structure de géodésie

#define NOM_FICH_BIS 256
#define PX_SIZE 0.5
#define NCC_THRESHOLD 0.7 

typedef struct{
  Complex8 ncc_vector;
  Complex8 uml_vector;
  Complex8 gnd_vector;
  Complex8 sinc_vector;
  Complex8 offset_vector;
  Complex8 offset_simu_vector;
  Complex8 corrected_ncc_vector;
  Complex8 corrected_uml_vector;
  Complex8 corrected_sinc_vector;
  Complex8 correc_simu_ncc_vector;
  Complex8 correc_simu_uml_vector;
  Complex8 correc_simu_sinc_vector;
  Complex8 error_ncc_vector;
  Complex8 error_uml_vector;
  Complex8 error_sinc_vector;
  Complex8 error_simu_ncc_vector;
  Complex8 error_simu_uml_vector;
  Complex8 error_simu_sinc_vector;
}Vec_gnd_sar; //Re: dx (range) - Im: dy (azimuth)

typedef struct{
  float east; // ground east component of displacement
  float north; // ground north component of displacement
  float up; // ground up component of displacement
  double longitude; // latitude coordinate
  double latitude; // longitude coordinate
}Vec_3d_gnd;

typedef struct{
  float magnitude;
  float orientation;
}Magni_orien;

/**
 * \ingroup compare_ground_sar
 * \typedef Compare_ground_sar_param
 *
 * \brief reference on compare_ground_sar_param structure
 *
 */
/**
 * \ingroup compare_ground_sar
 * \struct compare_ground_sar_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_compare_ground_sar_param{
  float alpha; /*!< inclinaison angle of satellite orbit */
  float theta; /*!< incidence angle of satellite */
  int track; /*!< choice ascending or descending track */
  float delta_temporal; /*!< temporal baseline between 2 images */
  float range_spacing; /*!< range pixel spacing */
  float azimuth_spacing; /*!< azimuth pixel spacing */
  int range_position; /*!< range position (column) of comparison point */
  int azimuth_position; /*!< azimuth position (line) of comparison point */
  int window_columns; /*!< number of columns (range) of average vector window */
  int window_lines; /*!< number of lines (azimuth) of average vector window */
  int ground_compare_choice; /*!< choice of ground measure type for comparison with remote sensing measure */
  int recal_value; /*!< value of remaining coregistration*/
}compare_ground_sar_param, *Compare_ground_sar_param;

/**
 * \ingroup compare_ground_sar
 * \fn new_compare_ground_sar_param()
 *
 * \return A new reference on allocated compare_ground_sar_param structure
 *
 * \brief Create a new Compare_ground_sar_param reference 
 *
 */
static Compare_ground_sar_param new_compare_ground_sar_param();
/**
 * \ingroup compare_ground_sar
 * \fn free_compare_ground_sar_param(Compare_ground_sar_param p)
 *
 * \param p A reference on a compare_ground_sar_param structure
 *
 * \brief Free an allocated Compare_ground_sar_param reference 
 *
 */
void free_compare_ground_sar_param(Compare_ground_sar_param p);


/**
 * \ingroup compare_ground_sar
 * \fn void define_compare_ground_sar_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_compare_ground_sar_param(char *extra_description);

/**
 * \ingroup compare_ground_sar
 * \fn Compare_ground_sar_param get_compare_ground_sar_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Compare_ground_sar_param that contains the parameters.
 */
Compare_ground_sar_param get_compare_ground_sar_param();

/**
 * \ingroup compare_ground_sar
 * \fn void compare_ground_sar(Compare_ground_sar_param p,char *data_name,char *georef_name,char *nccImage,char *umlImage, EFIDIRImage inputImage_mask, EFIDIRImage inputImage_offset,char *correc_path);
 * \brief function that compares ground and sar displacement measurement
 * Function comparing ground displacement measurement with SAR displacement measurement. This comparison is made in SAR geometry. Two types of comparison is available. 1) Comparaison between GPS and artificial target tracking (sinc, NCC and UML). 2) Comparison between stick and textured part of glacier (NCC and UML).
 *
 * \param p is the structure parameter for the operator
 * \param data_name is the GPS data file name 
 * \param georef_name is the georef data file name for geocoded residual offset simulation image
 * \param nccImage is a NCC image file name (_VAL and _VEC)
 * \param umlImage is a UML image file name (_VAL and _VEC)
 * \param inputImage_mask is mask image to isolate fixed zones
 * \param inputImage_offset is residual offset simulation image
 * \param correc_path is the path of displacement corrected image
 */
void compare_ground_sar(Compare_ground_sar_param p,char *data_name,char *georef_name,char *nccImage,char *umlImage, EFIDIRImage inputImage_mask, EFIDIRImage inputImage_offset,char *correc_path);

/**
 * \ingroup compare_ground_sar
 * \fn void ground_to_sar(Compare_ground_sar_param p, char *data_name, Vec_gnd_sar *vec_2d, Vec_3d_gnd *vec_3d);
 * \brief function that converts ground measurements to sar geometry
 * Function converting ground measurements (displacement) into SAR geometry.
 *
 * \param p is the structure parameter for the operator
 * \param data_name is the GPS data file name
 * \param vec_2d is the sar projection vector
 * \param vec_3d is the ground projection vector
 */
static void ground_to_sar(Compare_ground_sar_param p, char *data_name, Vec_gnd_sar *vec_2d, Vec_3d_gnd *vec_3d);

static void read_gps_sinc_dat(char *data_name,Vec_3d_gnd *vec_3d,Complex8 *sinc);

static void cal_magnitude_orientation(Complex8 vectors2D,Magni_orien *dpl_ori);

static int read_georef(char *nom_fichier_georef, TMapInfo *map_info);

//static void write_sar_dpl_dat(Compare_ground_sar_param p,char *name,float los, float track);

#endif /* LAR_COMPARE_GROUND_SAR_H */
