/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup interferometry
 * \defgroup varphi Estimation of variance of phase value from coherence
 */
/**
 * \ingroup varphi
 * \file iff_varphi.h
 * \author Yajing Yan
 * \brief operator of estimation of variance of phase value from coherence
 * 
 * file includes (stucture, prototype de fonction) of the operator varphi
 * 
 * 
 */

#ifndef IFF_VARPHI_H
#define IFF_VARPHI_H

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"


/**
 * \ingroup varphi
 * \typedef Varphi_param
 *
 * \brief reference on varphi_param structure
 *
 */
/**
 * \ingroup varphi
 * \struct varphi_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_varphi_param{
  int look; /*!< number of multi-looking */
  float lambda; /*!< wavelength */
}varphi_param, *Varphi_param;

/**
 * \ingroup varphi
 * \fn Varphi_param new_varphi_param()
 *
 * \return A new reference on allocated varphi_param structure
 *
 * \brief Create a new Varphi_param reference 
 *
 */
static Varphi_param new_varphi_param();
/**
 * \ingroup varphi
 * \fn void free_varphi_param(Varphi_param p)
 *
 * \param p A reference on a varphi_param structure
 *
 * \brief Free an allocated Varphi_param reference 
 *
 */
void free_varphi_param(Varphi_param p);


/**
 * \ingroup varphi
 * \fn void define_varphi_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_varphi_param(char *extra_description);

/**
 * \ingroup varphi
 * \fn Varphi_param get_varphi_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Varphi_param that contains the parameters.
 */
Varphi_param get_varphi_param();

/**
 * \ingroup varphi
 * \fn void varphi(Varphi_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief function of estimation of phase variance
 * Function allowing to estimate phase variance from coherence
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the coherence image
 * \param outputImage is the phase variance
 *
 */
void varphi(Varphi_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);


#endif
