/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup divers
 * \defgroup variogram Generation of variogram
 */
/**
 * \ingroup variogram
 * \file iff_variogram.h
 * \author Yajing Yan
 * 
 * \brief operator of generation of variogram
 * 
 * file includes (stucture, prototype de fonction) of the operator variogram
 * 
 * 
 */

#ifndef IFF_VARIOGRAM_H
#define IFF_VARIOGRAM_H

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "com_statistics.h"
#include "efidir_math.h"

/**
 * \ingroup variogram
 * \typedef Variogram_param
 *
 * \brief reference on variogram_param structure
 *
 */
/**
 * \ingroup variogram
 * \struct variogram_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_variogram_param{
  int index; /*!< index of variogram generation processing */
  int size; /*!< size of sliding window */
}variogram_param, *Variogram_param;

/**
 * \ingroup variogram
 * \fn Variogram_param new_variogram_param()
 *
 * \return A new reference on allocated variogram_param structure
 *
 * \brief Create a new Variogram_param reference 
 *
 */
static Variogram_param new_variogram_param();
/**
 * \ingroup variogram
 * \fn void free_variogram_param(Variogram_param p)
 *
 * \param p A reference on a variogram_param structure
 *
 * \brief Free an allocated Variogram_param reference 
 *
 */
void free_variogram_param(Variogram_param p);


/**
 * \ingroup variogram
 * \fn void define_variogram_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_variogram_param(char *extra_description);

/**
 * \ingroup variogram
 * \fn Variogram_param get_variogram_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Variogram_param that contains the parameters.
 */
Variogram_param get_variogram_param();

/**
 * \ingroup variogram
 * \fn void variogram(float **data, int x, int y, int nx, int ny, int h, int index, double var, double cov, double e);
 * \brief function of generation of variogram
 * Function allowing to generate a variogram
 *
 * \param data is the data for calculation of variance and covariance
 * \param x is the coordonate x of point 
 * \param y is the coordonate y of point 
 * \param nx is the number of points to calculate in x
 * \param ny is the number of points to calculate in y
 * \param h is the distance between two points
 * \param index is the index of processing (the direction of h)
 * \param var is the variance value
 * \param cov is the covariance value
 * \param e is a limite value
 *
 */
void variogram(float **data, int x, int y, int nx, int ny, int h, int index, double var, double cov, double e);


/**
 * \ingroup variogram
 * \fn void variance(Variogram_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage outputImage, EFIDIRImage outputMask);
 * \brief function of computation of variance
 * Function allowing to calculate the variance for each pixel of image
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImage is the input image
 * \param inputMask is the mask of input image
 * \param outputImage is the output image
 * \param outputMask is the mask of output image
 *
 */
void variance(Variogram_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage outputImage, EFIDIRImage outputMask);

#endif
