/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup interferometry
 * \defgroup unwrap  Interferometric phase unwrapping
 */

/**
 * \ingroup unwrap
 * \file iff_unwrap.h
 * \author Yajing Yan and Emmanuel Trouve
 * 
 * \brief A method for phase unwrapping
 * 
 * file includes (stucture, prototype de fonction) of the estimator unwrap
 * 
 * 
 */

#ifndef IFF_UNWRAP_H
#define IFF_UNWRAP_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

//ET #include <nr.h>
#include <fftw3.h>

#include "efidir_math.h"
#include "efidir_image.h"
#include "efidir_param.h"

/**
 * \ingroup unwrap
 * \typedef Unwrap_param
 *
 * \brief reference on unwrap_param structure
 *
 */
/**
 * \ingroup unwrap
 * \struct unwrap_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_unwrap_param
{
    /* parameters */
    int Nb_iter;        /*!< maximum iteration number */
    int t;     /*!< threshold between 0 and 255 for binary weighting.-1:fuzzy weighting [0, 1] */ 
    double eps;      /*!< critical value : |rk|/|r0| < eps */
    int rfirst; /*!< first line to unwrap */
    int cfirst; /*!< first column to unwrap */
    int rlast; /*!< last line to unwrap */
    int clast; /*!< last column to unwrap */
}unwrap_param, *Unwrap_param;

/**
 * \ingroup unwrap
 * \fn new_unwrap_param()
 *
 * \return A new reference on allocated unwrap_param structure
 *
 * \brief Create a new Unwrap_param reference 
 *
 */
static Unwrap_param new_unwrap_param();
/**
 * \ingroup unwrap
 * \fn free_unwrap_param(Unwrap_param p)
 *
 * \param p A reference on a unwrap_param structure
 *
 * \brief Free an allocated Unwrap_param reference 
 *
 */
void free_unwrap_param(Unwrap_param p);


/**
 * \ingroup unwrap
 * \fn void define_unwrap_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_unwrap_param(char *extra_description);

/**
 * \ingroup unwrap
 * \fn Unwrap_param get_unwrap_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Unwrap_param that contains the parameters.
 */
Unwrap_param get_unwrap_param();


/**
 * \ingroup unwrap
 * \fn void poisson (EFIDIRImage res, EFIDIRImage data, EFIDIRImage work, double *tabcosX, double *tabcosY);
 * \brief solution of equation of poisson using cosinus transform
 *
 * \param res is the image output 
 * \param data is the input image 
 * \param work is a buffer image
 * \param tabcosX is the cos table in X
 * \param tabcosY is the cos table in Y
 *
 */
void poisson (EFIDIRImage res, EFIDIRImage data, EFIDIRImage work, double *tabcosX, double *tabcosY);

/**
 * \ingroup unwrap
 * \fn float v_scal_v (float *v1, float *v2, int d);
 * \brief scalar product of two float vectors
 *
 * \param v1 is the first float vector
 * \param v2 is the second float vector
 * \param d is the dimension of vectors
 * \return A float number that results from the scalar product of two float vectors 
 *
 */
 float v_scal_v (float *v1, float *v2, int d);

/**
 * \ingroup unwrap
 * \fn void kv_plus_v (float *v1, float k, float *v2, int d);
 * \brief calculate k*v1+v2
 *
 * \param v1 is the frist vector
 * \param k is the coefficient to be mulplicated by v1
 * \param v2 is the second vector
 * \param d is the dimension of the two vectors
 *
 */
 void kv_plus_v (float *v1, float k, float *v2, int d);

/**
 * \ingroup unwrap
 * \fn void v_plus_kv (float *v1, float *v2, float k, int d);
 * \brief calculate v1+k*v2
 *
 * \param v1 is the frist vector
 * \param v2 is the second vector
 * \param k is the coefficient to be mulplicated by v2
 * \param d is the dimension of the two vectors
 *
 */
 void v_plus_kv (float *v1, float *v2, float k, int d);

/**
 * \ingroup unwrap
 * \fn void lapla_weight (EFIDIRImage res, EFIDIRImage data, EFIDIRImage poid);
 * \brief calculate laplacien weight
 *
 * \param res is the image to output
 * \param data is a buffer image
 * \param poid is the confidence image
 *
 */
void lapla_weight (EFIDIRImage res, EFIDIRImage data, EFIDIRImage poid);

/**
 * \ingroup unwrap
 * \fn void unwrap(Unwrap_param p, EFIDIRImage inputFrx, EFIDIRImage inputFry,EFIDIRImage inputConf, EFIDIRImage outputImage, EFIDIRImage outputmask);
 * \brief function of phase unwrapping
 * Function for phase unwrapping by a least square method
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputFrx is the estimated local frequency in x
 * \param inputFry is the estimated local frequency in y
 * \param inputConf is the confidence index image
 * \param outputImage is the result of unwrap operation
 * \param outputmask is the mask of measurability
 *
 */
void unwrap(Unwrap_param p, EFIDIRImage inputFrx, EFIDIRImage inputFry,EFIDIRImage inputConf, EFIDIRImage outputImage, EFIDIRImage outputmask);


#endif
