/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup 3D_inversion
 * \defgroup surfdisp Surface 3D displacement estimation
 */
/**
 * \ingroup surfdisp
 * \file iff_surfdisp.h
 * \author Yajing Yan
 * 
 * \brief operator of calculating the surface displacement in a (E,N,U) framework from the displacements of different projections
 * 
 * file includes (stucture, prototype de fonction) of the operator surfdisp
 * 
 * 
 */

#ifndef IFF_SURFDISP_H
#define IFF_SURFDISP_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include "efidir_matrix_op.h"


/**
 * \ingroup surfdisp
 * \typedef Surfdisp_param
 *
 * \brief reference on surfdisp_param structure
 *
 */
/**
 * \ingroup surfdisp
 * \struct surfdisp_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_surfdisp_param
{
  int process; /*!< processing index */
  float X_min; /*!< first coordonate in X to process */
  float Y_min; /*!< first coordonate in Y to process */
  float X_max; /*!< last coordonate in X to process */
  float Y_max; /*!< last  coordonate in Y to process */
  int nb; /*!< number of projections */
}surfdisp_param, *Surfdisp_param;

/**
 * \ingroup surfdisp
 * \fn new_surfdisp_param()
 *
 * \return A new reference on allocated surfdisp_param structure
 *
 * \brief Create a new Surfdisp_param reference 
 *
 */
static Surfdisp_param new_surfdisp_param();
/**
 * \ingroup surfdisp
 * \fn free_surfdisp_param(Surfdisp_param p)
 *
 * \param p A reference on a surfdisp_param structure
 *
 * \brief Free an allocated Surfdisp_param reference 
 *
 */
void free_surfdisp_param(Surfdisp_param p);


/**
 * \ingroup surfdisp
 * \fn void define_surfdisp_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_surfdisp_param(char *extra_description);

/**
 * \ingroup surfdisp
 * \fn Surfdisp_param get_surfdisp_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Surfdisp_param that contains the parameters.
 */
Surfdisp_param get_surfdisp_param();

/**
 * \ingroup surfdisp
 * \fn void surfdisp(Surfdisp_param p, EFIDIRImage *def, EFIDIRImage *mask, EFIDIRImage *error, EFIDIRImage *u, EFIDIRImage *Sigmau);
 * \brief function of calculating the surface displacement from different projections
 * Function allowing to calculate the surface displacement in the frameword (E,N,U) from the displacements of different 
 * projections.
 *
 * \param p is the structure of the parameters for the operator 
 * \param def is the different projections of deformation
 * \param mask is the masks of different projections of deformation
 * \param error is the uncertainty image of different projections of deformation
 * \param u is the surface displacements in the framework (E, N, UP)
 * \param Sigmau is the strandard deviation of surface displacement
 * 
 */
void surfdisp(Surfdisp_param p, EFIDIRImage *def, EFIDIRImage *mask, EFIDIRImage *error, EFIDIRImage *u,  EFIDIRImage *Sigmau);

#endif
