/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup 3D_inversion
 * \defgroup select_surfdisp Surface 3D displacement estimation with selection of projections
 */
/**
 * \ingroup select_surfdisp
 * \file iff_select_surfdisp.h
 * \author Yajing Yan
 * 
 * \brief operator of calculating the surface displacement in a (E,N,U) framework from 4 selected displacements of different projections
 * 
 * file includes (stucture, prototype de fonction) of the operator select_surfdisp
 * 
 * 
 */

#ifndef IFF_SELECT_SURFDISP_H
#define IFF_SELECT_SURFDISP_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include "efidir_matrix_op.h"


/**
 * \ingroup select_surfdisp
 * \typedef Select_surfdisp_param
 *
 * \brief reference on select_surfdisp_param structure
 *
 */
/**
 * \ingroup select_surfdisp
 * \struct select_surfdisp_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_select_surfdisp_param
{
  int process; /*!< processing index */
  float X_min; /*!< first coordonate in X to process */
  float Y_min; /*!< first coordonate in Y to process */
  float X_max; /*!< last coordonate in X to process */
  float Y_max; /*!< last  coordonate in Y to process */
  int nb; /*!< number of projections */
}select_surfdisp_param, *Select_surfdisp_param;

/**
 * \ingroup select_surfdisp
 * \fn new_select_surfdisp_param()
 *
 * \return A new reference on allocated select_surfdisp_param structure
 *
 * \brief Create a new Select_surfdisp_param reference 
 *
 */
static Select_surfdisp_param new_select_surfdisp_param();
/**
 * \ingroup select_surfdisp
 * \fn free_select_surfdisp_param(Select_surfdisp_param p)
 *
 * \param p A reference on a select_surfdisp_param structure
 *
 * \brief Free an allocated Select_surfdisp_param reference 
 *
 */
void free_select_surfdisp_param(Select_surfdisp_param p);


/**
 * \ingroup select_surfdisp
 * \fn void define_select_surfdisp_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_select_surfdisp_param(char *extra_description);

/**
 * \ingroup select_surfdisp
 * \fn Select_surfdisp_param get_select_surfdisp_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Select_surfdisp_param that contains the parameters.
 */
Select_surfdisp_param get_select_surfdisp_param();

/**
 * \ingroup select_surfdisp
 * \fn void select_surfdisp(Select_surfdisp_param p, EFIDIRImage *def, EFIDIRImage *mask, EFIDIRImage *error, EFIDIRImage *u, EFIDIRImage *Sigmau, EFIDIRImage *info);
 * \brief function of calculating the surface displacement from different projections
 * Function allowing to calculate the surface displacement in the frameword (E,N,U) from the displacements of different 
 * projections.
 *
 * \param p is the structure of the parameters for the operator 
 * \param def is the different projections of deformation
 * \param mask is the masks of different projections of deformation
 * \param error is the uncertainty image of different projections of deformation
 * \param u is the surface displacements in the framework (E, N, UP)
 * \param Sigmau is the standard deviation of surface displacement
 * \param info is the information image indicating the used pixels of each projection image
 * 
 */
void select_surfdisp(Select_surfdisp_param p, EFIDIRImage *def, EFIDIRImage *mask, EFIDIRImage *error, EFIDIRImage *u, EFIDIRImage *Sigmau, EFIDIRImage *info);

#endif
