/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup interferometry
 * \defgroup sar_scale Determination of scale for phase unwrapping
 */
/**
 * \ingroup sar_scale
 * \file iff_sar_scale.h
 * \author Yajing Yan (LISTIC, LGIT)
 * 
 * \brief Determination of scale for phase unwrapping according to the gradient of deformation
 * 
 * file includes (stucture, prototype de fonction) of the operator sar_scale
 * 
 * 
 */

#ifndef IFF_SAR_SCALE_H
#define IFF_SAR_SCALE_H

#include <stdio.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"


/**
 * \ingroup sar_scale
 * \typedef Sar_scale_param
 *
 * \brief reference on sar_scale_param structure
 *
 */
/**
 * \ingroup sar_scale
 * \struct sar_scale_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_sar_scale_param{
  int index; /*!< index of processing */
  float lambda; /*!< wavelength of radar */
}sar_scale_param, *Sar_scale_param;

/**
 * \ingroup sar_scale
 * \fn new_sar_scale_param()
 *
 * \return A new reference on allocated sar_scale_param structure
 *
 * \brief Create a new Sar_scale_param reference 
 *
 */
static Sar_scale_param new_sar_scale_param();
/**
 * \ingroup sar_scale
 * \fn free_sar_scale_param(Sar_scale_param p)
 *
 * \param p A reference on a sar_scale_param structure
 *
 * \brief Free an allocated Sar_scale_param reference 
 *
 */
void free_sar_scale_param(Sar_scale_param p);


/**
 * \ingroup sar_scale
 * \fn void define_sar_scale_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_sar_scale_param(char *extra_description);

/**
 * \ingroup sar_scale
 * \fn Sar_scale_param get_sar_scale_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Sar_scale_param that contains the parameters.
 */
Sar_scale_param get_sar_scale_param();

/**
 * \ingroup sar_scale
 * \fn void sar_scale(Sar_scale_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief function of determination of scale for phase unwrapping
 * Function allowing to calculate the appropriate scale for phase unwrapping from a deformation model
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the image of deformation model
 * \param outputImage is the image containing scale information for each pixel
 *
 */
void sar_scale(Sar_scale_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);


#endif
