/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup divers
 * \defgroup regression_2D Simulation of estimation of deformation without phase unwrapping
 */
/**
 * \ingroup regression_2D
 * \file iff_regression_2D.h
 * \author Yajing Yan and Marie-pierre Doin
 * 
 * \brief simulation of estimation of deformation without phase unwrapping
 * 
 * file includes (stucture, prototype de fonction) of the operator regression_2D
 * 
 * 
 */

#ifndef IFF_REGRESSION_2D_H
#define IFF_REGRESSION_2D_H

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "com_wrap.h"

/**
 * \ingroup regression_2D
 * \typedef Regression_2D_param
 *
 * \brief reference on regression_2D_param structure
 *
 */
/**
 * \ingroup regression_2D
 * \struct regression_2D_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_regression_2D_param{
  int index; /*!< index of regression_2Dlation processing */
  int nb; /*!< number of interferograms in the temporal series */
  int ninterval; /*!< number of temporal interval */
  int nnoise; /*!< number of noise interval */
  float vmax; /*!< maximum value of deformation velocity */
  float vmin; /*!< minimum value of deformation velocity */
  float a; /*!< acceleration of deformation */
  float vinput; /*!< deformation veloctiy */
  float hmax; /*!< maximum value of elevation correction */
  float hmin;  /*!< minimum value of elevation correction */
  float hinput; /*!< elevation correction */
  float scale; /*!< scale of calculation */
  float angle; /*!< incidence angle */
  float dist; /*!< distance between radar and the target */
  float amp; /*!< standard deviation of noise */
  float period; /*!< period of sinusoid deformation */
}regression_2D_param, *Regression_2D_param;

/**
 * \ingroup regression_2D
 * \fn new_regression_2D_param()
 *
 * \return A new reference on allocated regression_2D_param structure
 *
 * \brief Create a new regression_2D_param reference 
 *
 */
static Regression_2D_param new_regression_2D_param();
/**
 * \ingroup regression_2D
 * \fn static void free_regression_2D_param(Regression_2D_param p);
 *
 * \param p A reference on a regression_2D_param structure
 *
 * \brief Free an allocated regression_2D_param reference 
 *
 */
void free_regression_2D_param(Regression_2D_param p);


/**
 * \ingroup regression_2D
 * \fn void define_regression_2D_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_regression_2D_param(char *extra_description);

/**
 * \ingroup regression_2D
 * \fn Regression_2D_param get_regression_2D_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A regression_2D_param that contains the parameters.
 */
Regression_2D_param get_regression_2D_param();

/**
 * \ingroup regression_2D
 * \fn void regression_2D(Regression_2D_param p, char *date_file, char *base_file, char *output_vfile, char *output_hfile, char *output_file);
 * \brief function of regression_2Dlation of estimation of deformation without phase unwrapping
 * Function allowing to regression_2Dlate the processing of estimation of deformation velocity without phase unwrapping
 * using a method of 2D regression from a series of interferograms
 *
 * \param p is the structure of the parameters for the operator 
 * \param date_file is the file containing the temporal interval
 * \param base_file is the file containing the perpenticular baseline
 * \param output_vfile is the file containing the velocity 
 * \param output_hfile is the file containing the elevation correction
 * \param output_file is the file containing the coherence
 *
 */
void regression_2D(Regression_2D_param p, char *date_file, char *base_file, char *output_vfile, char *output_hfile, char *output_file);


#endif
