/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup model_inversion
 * \defgroup quadtree Quadtree Resampling
 */
/**
 * \ingroup quadtree
 * \file iff_quadtree.h
 * \author Yajing Yan (LISTIC,LGIT)
 * 
 * \brief Quadtree resampling
 * 
 * file includes (stucture, prototype de fonction) of the operator quatree
 * 
 * 
 */

#ifndef IFF_QUADTREE_H
#define IFF_QUADTREE_H

#include <stdio.h>
#include<stdlib.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "efidir_string.h"
#include "com_filter_median.h"

/**
 * \ingroup quadtree
 * \typedef Quadtree_param
 *
 * \brief reference on quadtree_param structure
 *
 */

/**
 * \ingroup quadtree
 * \struct cell
 *
 * \brief structure that contains the coordinate of quadtree square
 *
 */
typedef struct cell{
    int Xmin;
    int Xmax;
    int Ymin;
    int Ymax;
}cell, *Cell;

//typedef cell DataType;

/**
 * \ingroup quadtree
 * \struct qnode
 *
 * \brief structure that contains the node of quadtree square
 *
 */
typedef struct qnode{
   cell data;
   struct qnode *next;
}qnode, *Qnode;

/**
 * \ingroup quadtree
 * \struct link_queue
 *
 * \brief structure that contains the pointers of queue
 *
 */

typedef struct link_queue{
   Qnode front;
   Qnode rear;
}link_queue, *Link_queue;

/**
 * \ingroup quadtree
 * \struct quadtree_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_quadtree_param{
 float threshold; /*!< threshold for split */
}quadtree_param, *Quadtree_param;


/**
 * \ingroup quadtree
 * \fn void InitQueue(struct link_queue *Q);
 *
 * \param Q is the link_queue pointer
 *
 * \brief Initiate a queue FIFO
 *
 */
void InitQueue(struct link_queue *Q);

/**
 * \ingroup quadtree
 * \fn void EnQueue(struct link_queue *Q, cell c);
 *
 * \param Q is the link_queue pointer
 * \param c is the element to be added in the queue
 *
 * \brief Add a new element in the queue
 *
 */
void EnQueue(struct link_queue *Q, cell c);

/**
 * \ingroup quadtree
 * \fn cell DeQueue(struct link_queue *Q);
 *
 * \param Q is the link_queue pointer
 * \return the deleted element
 *
 * \brief Delete an element in the queue
 *
 */
cell DeQueue(struct link_queue *Q);

/**
 * \ingroup quadtree
 * \fn int QueueEmpty(struct link_queue *Q);
 *
 * \param Q is the link_queue pointer
 * \return 1 if the queue is empty, 0 if not
 *
 * \brief Verify if the queue is empty
 *
 */
int QueueEmpty(struct link_queue *Q);

/**
 * \ingroup quadtree
 * \fn void DestroyQueue(struct link_queue *Q);
 *
 * \param Q is the link_queue pointer
 *
 * \brief Destroy the queue
 *
 */
void DestroyQueue(struct link_queue *Q);

/**
 * \ingroup quadtree
 * \fn cell QueueFront(struct link_queue *Q);
 *
 * \param Q is the link_queue pointer
 * \return the frist element in the queue
 *
 * \brief Get the first element in the queue
 *
 */
cell QueueFront(struct link_queue *Q);


/**
 * \ingroup quadtree
 * \fn new_quadtree_param()
 *
 * \return A new reference on allocated quadtree_param structure
 *
 * \brief Create a new Quadtree_param reference
 *
 */
static Quadtree_param new_quadtree_param();

/**
 * \ingroup quadtree
 * \fn free_quadtree_param(Quadtree_param p)
 *
 * \param p A reference on a quadtree_param structure
 *
 * \brief Free an allocated Quadtree_param reference
 *
 */
void free_quadtree_param(Quadtree_param p);

/**
 * \ingroup quadtree
 * \fn void define_quadtree_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_quadtree_param(char *extra_description);

/**
 * \ingroup quadtree
 * \fn Quadtree_param get_quadtree_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Quadtree_param that contains the parameters.
 */
Quadtree_param get_quadtree_param();

/**
 * \ingroup quadtree
 * \fn void quadtree(Quadtree_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage inputError, char *filename, char *locationfile);
 * \brief function of quadtree resampling
 * Function allowing to sub-resample an image
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImage is the image to resample
 * \param inputMask is the mask of input image
 * \param inputError is the associated error image
 * \param filename is the output ASCII file 
 * \param locationfile is the quadtree rectangle location file
 *
 */

void quadtree(Quadtree_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage inputError, char *filename, char *locationfile);

#endif
