/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup projection
 * \defgroup projection_vector Calculation of projection vector
 */
/**
 * \ingroup projection_vector
 * \file iff_projection_vector.h
 * \author Yajing Yan
 * 
 * \brief operator of generation of projection_vector
 * 
 * file includes (stucture, prototype de fonction) of the operator projection_vector
 * 
 * 
 */

#ifndef IFF_PROJECTION_VECTOR_H
#define IFF_PROJECTION_VECTOR_H

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "efidir_string.h"

/**
 * \ingroup projection_vector
 * \typedef Projection_vector_param
 *
 * \brief reference on projection_vector_param structure
 *
 */
/**
 * \ingroup projection_vector
 * \struct projection_vector_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_projection_vector_param{
  int mode; /*!< mode of acquisition */
  int rfirst; /*!< first line to calculate */
  int cfirst; /*!< first column to calculate */
  int rlast; /*!< last line to calculate */
  int clast; /*!< last column to calculate */
}projection_vector_param, *Projection_vector_param;

/**
 * \ingroup projection_vector
 * \fn new_projection_vector_param()
 *
 * \return A new reference on allocated projection_vector_param structure
 *
 * \brief Create a new Projection_vector_param reference 
 *
 */
static Projection_vector_param new_projection_vector_param();
/**
 * \ingroup projection_vector
 * \fn free_projection_vector_param(Projection_vector_param p)
 *
 * \param p A reference on a projection_vector_param structure
 *
 * \brief Free an allocated Projection_vector_param reference 
 *
 */
void free_projection_vector_param(Projection_vector_param p);


/**
 * \ingroup projection_vector
 * \fn void define_projection_vector_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_projection_vector_param(char *extra_description);

/**
 * \ingroup projection_vector
 * \fn Projection_vector_param get_projection_vector_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Projection_vector_param that contains the parameters.
 */
Projection_vector_param get_projection_vector_param();

/**
 * \ingroup mean_vector
 * \fn void mean_vector(Projection_vector_param p, EFIDIRImage inputImage);
 * \brief function of generation of an average projection vector
 * Function allowing to generate an average projection vector 
 * The projection of deformation (E,N,UP) in the direction of range and azimuth
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the image of incidence angle
 *
 */
void mean_vector(Projection_vector_param p, EFIDIRImage inputImage);



/**
 * \ingroup projection_vector
 * \fn void projection_vector(Projection_vector_param p, EFIDIRImage inputImage, EFIDIRImage outputImageE_rg, EFIDIRImage outputImageN_rg, EFIDIRImage outputImageUp_rg, EFIDIRImage outputImageE_az, EFIDIRImage outputImageN_az, EFIDIRImage outputImageUp_az);
 * \brief function of generation of projection_vector
 * Function allowing to generate a projection_vector according to the value of image
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the image of incidence angle
 * \param outputImageE_rg is the projection of Est component in range
 * \param outputImageN_rg is the projection of North component in range
 * \param outputImageUp_rg is the projection of Up component in range
 * \param outputImageE_az is the projection of Est component in azimuth
 * \param outputImageN_az is the projection of North component in azimuth
 * \param outputImageUp_az is the projection of Up component in azimuth
 *
 */
void projection_vector(Projection_vector_param p, EFIDIRImage inputImage, EFIDIRImage outputImageE_rg, EFIDIRImage outputImageN_rg, EFIDIRImage outputImageUp_rg, EFIDIRImage outputImageE_az, EFIDIRImage outputImageN_az, EFIDIRImage outputImageUp_az);

#endif
