/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup divers
 * \defgroup profile profile generation
 */
/**
 * \ingroup profile
 * \file iff_profile.h
 * \author Yajing Yan
 * 
 * \brief operator of generation of a profile
 * 
 * file includes (stucture, prototype de fonction) of the operator profile
 * 
 * 
 */

#ifndef IFF_PROFILE_H
#define IFF_PROFILE_H

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "com_rotate.h"

/**
 * \ingroup profile
 * \typedef Profile_param
 *
 * \brief reference on profile_param structure
 *
 */
/**
 * \ingroup profile
 * \struct profile_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_profile_param{
  float a; /*!< parametre of profile line y=ax+b */
  float b; /*!< parametre of profile line y=ax+b */
  int refx; /*!< column position of reference point */
  int refy; /*!< line position of reference point */
  float threshold; /*!< threshold */
  int index; /*!< processing index */
  int width; /*!< width of profile band */
}profile_param, *Profile_param;

/**
 * \ingroup profile
 * \fn Profile_param new_profile_param()
 *
 * \return A new reference on allocated profile_param structure
 *
 * \brief Create a new Profile_param reference 
 *
 */
static Profile_param new_profile_param();
/**
 * \ingroup profile
 * \fn void free_profile_param(Profile_param p)
 *
 * \param p A reference on a profile_param structure
 *
 * \brief Free an allocated Profile_param reference 
 *
 */
void free_profile_param(Profile_param p);


/**
 * \ingroup profile
 * \fn void define_profile_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_profile_param(char *extra_description);

/**
 * \ingroup profile
 * \fn Profile_param get_profile_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Profile_param that contains the parameters.
 */
Profile_param get_profile_param();

/**
 * \ingroup profile
 * \fn void disp_profile(Profile_param p, EFIDIRImage inputImage, char *file);
 * \brief function of generation of a profile
 * Function allowing to generate a profile
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the input image
 * \param file is a text file containing the parameters for the profile
 *
 */
void disp_profile(Profile_param p, EFIDIRImage inputImage, char *file);


/**
 * \ingroup profile
 * \fn void mean_profile(Profile_param p, EFIDIRImage inputImage, char *file);
 * \brief function of generation of an averaged profile 
 * Function allowing to generate a profile, the value for each point on the profile is the mean value of a profile band.
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the input image
 * \param file is a text file containing the parameters for the profile
 *
 */
void mean_profile(Profile_param p, EFIDIRImage inputImage, char *file);


#endif
