/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup divers
 * \defgroup noise_generator Generation of noise
 */
/**
 * \ingroup noise_generator
 * \file iff_noise_generator.h
 * \author Yajing Yan
 * 
 * \brief operator of noise generation
 * 
 * file includes (stucture, prototype de fonction) of the operator noise_generator
 * 
 * 
 */

#ifndef IFF_NOISE_GENERATOR_H
#define IFF_NOISE_GENERATOR_H

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "com_wrap.h"

/**
 * \ingroup noise_generator
 * \typedef Noise_generator_param
 *
 * \brief reference on noise_generator_param structure
 *
 */
/**
 * \ingroup noise_generator
 * \struct noise_generator_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_noise_generator_param{
  int index; /*!< index of noise_generator generation processing */
  int type; /*!< index of noise type */
  float sigma; /*!< variance of gaussian noise */
  float mean; /*!< mean value of gaussian noise */
  float sigma_min;  /*!< maximum variance of gaussian noise */
  float sigma_max; /*!< minimum variance of gaussian noise */
}noise_generator_param, *Noise_generator_param;

/**
 * \ingroup noise_generator
 * \fn Noise_generator_param new_noise_generator_param()
 *
 * \return A new reference on allocated noise_generator_param structure
 *
 * \brief Create a new Noise_generator_param reference 
 *
 */
static Noise_generator_param new_noise_generator_param();
/**
 * \ingroup noise_generator
 * \fn void free_noise_generator_param(Noise_generator_param p)
 *
 * \param p A reference on a noise_generator_param structure
 *
 * \brief Free an allocated Noise_generator_param reference 
 *
 */
void free_noise_generator_param(Noise_generator_param p);


/**
 * \ingroup noise_generator
 * \fn void define_noise_generator_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_noise_generator_param(char *extra_description);

/**
 * \ingroup noise_generator
 * \fn Noise_generator_param get_noise_generator_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Noise_generator_param that contains the parameters.
 */
Noise_generator_param get_noise_generator_param();

/**
 * \ingroup noise_generator
 * \fn float ranged_random_nb(float max, float min);
 * \brief function of generation of a ranged random number
 * Function allowing to generate a ranged random number
 *
 * \param max is the maximum value
 * \param min is the minimum value
 * \return nb is the generated random value between min and max
 *
 */
float ranged_random_nb(float max, float min);

/**
 * \ingroup noise_generator
 * \fn void noise_generator_image(Noise_generator_param p, EFIDIRImage inputImage, EFIDIRImage outputImage, EFIDIRImage noiseImage);
 * \brief function of noise generation on an image
 * Function allowing to generate noise (Gaussian, uniform) on an image
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the input image
 * \param outputImage is the output image
 * \param noiseImage is the image of noise
 *
 */
void noise_generator_image(Noise_generator_param p, EFIDIRImage inputImage, EFIDIRImage outputImage, EFIDIRImage noiseImage);


/**
 * \ingroup noise_generator
 * \fn void noise_generator_file(Noise_generator_param p, char *inputfile, char *outputfile, char *noisefile);
 * \brief function of noise generation in an ASCII file
 * Function allowing to generate noise (Gaussian, uniform) in an ASCII file
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputfile is the input file
 * \param outputfile is the output file
 * \param noisefile is the file of noise
 *
 */
void noise_generator_file(Noise_generator_param p, char *inputfile, char *outputfile, char *noisefile);


#endif
