/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup interferometry
 * \defgroup music Compute the LR local frequencies image (fx and fy) and the confidence image
 */
/**
 * \ingroup music
 * \file iff_music.h
 * \author Yajing Yan and Emmanuel Trouve
 * 
 * \brief operator of computing the LR local frequencies image (fx and fy) and the confidence image
 * 
 * file includes (stucture, prototype de fonction) of the operator music
 * 
 * 
 */

#ifndef IFF_MUSIC_H
#define IFF_MUSIC_H

#include <stdlib.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_math.h"

/**
 * \ingroup music
 * \typedef Music_param
 *
 * \brief reference on music_param structure
 *
 */

/**
 * \ingroup music
 * \struct music_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_music_param{
 /* parameters */
  int dimV;  /*!< size of vector signal window */
  int  dimF;/*!< size of covariance estimation window, inpair */
  int ch_er;    /*!< error ; Least square (0), eigenvalue (1), fusion of two (2)  */
  int first_line; /*!< fisrt line to calculate */
  int first_col; /*!< fisrt column to calculate */
  int last_line; /*!< last line to calculate */
  int last_col; /*!< last column to calculate */
/* buffer */
    double xxtR[256][256], xxtI[256][256];
    EFIDIRImage tmpR, tmpI;
    EFIDIRImage sum_covR, sum_covI;
    EFIDIRImage *pt_covR, *pt_covI;
    EFIDIRImage mod_cov;
    double *vectR, *vectI;
    double eMC;
}music_param, *Music_param;



/**
 * \ingroup music
 * \fn new_music_param()
 *
 * \return A new reference on allocated music_param structure
 *
 * \brief Create a new Music_param reference 
 *
 */
static Music_param new_music_param();

/**
 * \ingroup music
 * \fn free_music_param(Music_param p)
 *
 * \param p A reference on a music_param structure
 *
 * \brief Free an allocated Music_param reference 
 *
 */
void free_music_param(Music_param p);


/**
 * \ingroup music
 * \fn void define_music_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_music_param(char *extra_description);

/**
 * \ingroup music
 * \fn Music_param get_music_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Music_param that contains the parameters.
 */
Music_param get_music_param();

/**
 * \ingroup music
 * \fn void reset(EFIDIRImage inputImage1, EFIDIRImage inputImage2);
 * \brief reset an image
 * 
 * \param inputImage1 is the original image
 * \param inputImage2 is the resultant image
 *
 */
 void reset(EFIDIRImage inputImage1, EFIDIRImage inputImage2);

/**
 * \ingroup music
 * \fn void plusequal(EFIDIRImage Image1,EFIDIRImage Image2, EFIDIRImage Image3, EFIDIRImage Image4);
 * \brief intermediate calculations of images
 * 
 *
 * \param Image1 is the first image to input 
 * \param Image2 is the second image to input
 * \param Image3 is the first image to output
 * \param Image4 is the second image to output 
 */
void plusequal(EFIDIRImage Image1,EFIDIRImage Image2, EFIDIRImage Image3, EFIDIRImage Image4);

/**
 * \ingroup music
 * \fn void lessequal(EFIDIRImage Image1,EFIDIRImage Image2, EFIDIRImage Image3, EFIDIRImage Image4);
 * \brief intermediate calculations of images
 * 
 * \param Image1 is the first image to input 
 * \param Image2 is the second image to input
 * \param Image3 is the first image to output
 * \param Image4 is the second image to output
 */
void lessequal(EFIDIRImage Image1,EFIDIRImage Image2, EFIDIRImage Image3, EFIDIRImage Image4);

/**
 * \ingroup music
 * \fn void compl_ima(EFIDIRImage Image1,EFIDIRImage Image2);
 * \brief intermediate calculations of images
 *
 * \param Image1 is the first parameter image
 * \param Image2 is the second parameter image
 */
void compl_ima(EFIDIRImage Image1, EFIDIRImage Image2);

/**
 * \ingroup music
 * \fn void cov_col(Music_param p, EFIDIRImage inputPhase, int j0, int jM, int i0, EFIDIRImage pt_imaR, EFIDIRImage pt_imaI);
 * \brief intermediate calculations of images
 * 
 * \param p is the structure of the parameters for the operator 
 * \param inputPhase is the phase image
 * \param j0 is a parameter of function
 * \param jM is a parameter of function
 * \param i0 is a parameter of function
 * \param pt_imaR is the covariance of the real part
 * \param pt_imaI is the covariance of the imagenary part 
 */
void cov_col(Music_param p, EFIDIRImage inputPhase, int j0, int jM, int i0, EFIDIRImage pt_imaR, EFIDIRImage pt_imaI);

/**
 * \ingroup music
 * \fn Complex8 cov2freq(Music_param p,  EFIDIRImage coR, EFIDIRImage coI, Complex8 pfreq);
 * \brief calculation of covariance
 * 
 *
 * \param p is the structure of the parameters for the operator 
 * \param coR is the real part covariance
 * \param coI is the imagenary part covariance
 * \param pfreq is the local frequencies estimated for every pixel
 */
 Complex8 cov2freq(Music_param p, EFIDIRImage coR, EFIDIRImage coI, Complex8 pfreq);

/**
 * \ingroup music
 * \fn double proprete(Music_param p, double fx, double fy);
 * \brief calculation of eigenvector 
 * 
 * \param p is the structure of the parameters for the operator 
 * \param fx is  the local frequency in x
 * \param fy is the local frequency in y
 */
double proprete(Music_param p, double fx, double fy);

/**
 * \ingroup music
 * \fn void music(Music_param p, EFIDIRImage inputPhase, EFIDIRImage outputFreq, EFIDIRImage outputConf);
 * \brief function of computing the local frequencies and their confidence image
 * Function allowing to calculate the local frequencies (fx, fy) and their confidence image from the phase image
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputPhase is the phase image
 * \param outputFreq is the complex image of local frequencies (fx, fy)
 * \param outputConf is the confidence image of estiamtion
 *
 */
void music(Music_param p, EFIDIRImage inputPhase, EFIDIRImage outputFreq, EFIDIRImage outputConf);


#endif
