/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup interferometry
 * \defgroup local_fr Estimation of local frequency of interferometry phase
 */

/**
 * \ingroup local_fr
 * \file iff_local_fr.h
 * \author Yajing Yan, Gabriel Vasile, Emmanuel Trouve
 * 
 * \brief Estimator of local frequency of the interferometry phase image
 * 
 * file includes (stucture, prototype de fonction) of the estimator local_fr
 * 
 * 
 */

#ifndef IFF_LOCAL_FR_H
#define IFF_LOCAL_FR_H

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "efidir_image.h"
#include "efidir_param.h"

/**
 * \ingroup local_fr
 * \typedef Local_fr_param
 *
 * \brief reference on local_fr_param structure
 *
 */

/**
 * \ingroup local_fr
 * \struct coordonate
 *
 * \brief 
 *
 */
typedef struct struct_coordonate
{
 int m;
 int n;
} coordonate;

/**
 * \ingroup local_fr
 * \struct local_fr_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_local_fr_param
{
  /* parameters */
  int comp; /*!<LR compensation (0=true / 1=false) */
  double speckle_std; /*!< standard deviation of speckle */
  int MAX_REGION_SIZE; /*!< maximum size of the region */
  int h; /*!<height of phase image */
  int w; /*!< width of phase image */

  /* buffer */
  int ncf, nrf;
  unsigned int *map,**Pmap;
  unsigned region_count, background_count;
  unsigned int label;
  coordonate *region, *background;
  EFIDIRImage autocor_real, autocor_imag;
  EFIDIRImage n_points;
  EFIDIRImage tab_herm;
}local_fr_param, *Local_fr_param;

/**
 * \ingroup local_fr
 * \fn new_local_fr_param()
 *
 * \return A new reference on allocated local_fr_param structure
 *
 * \brief Create a new Local_fr_param reference 
 *
 */
static Local_fr_param new_local_fr_param();
/**
 * \ingroup local_fr
 * \fn free_local_fr_param(Local_fr_param p)
 *
 * \param p A reference on a local_fr_param structure
 *
 * \brief Free an allocated Local_fr_param reference 
 *
 */
void free_local_fr_param(Local_fr_param p);


/**
 * \ingroup local_fr
 * \fn void define_local_fr_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_local_fr_param(char *extra_description);

/**
 * \ingroup local_fr
 * \fn Local_fr_param get_local_fr_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Local_fr_param that contains the parameters.
 */
Local_fr_param get_local_fr_param();

/**
 * \ingroup local_fr
 * \fn double EstimateLocalMean(EFIDIRImage ima, int im, int jm, int iM, int jM);
 * \brief estimation of median value in an image
 *
 * \param ima is the image data
 * \param im is the first line to be considered
 * \param jm is the first column to be considered 
 * \param iM is the last line to be considered
 * \param jM is the last column to be considered
 * \return A median value of the part of image considered
 *
 */
double EstimateLocalMean(EFIDIRImage ima, int im, int jm, int iM, int jM);

/**
 * \ingroup local_fr
 * \fn void GrowRegion(Local_fr_param p, int m, int n, double *Pm1, double *Pm2, EFIDIRImage imamp1, EFIDIRImage imamp2);
 * \brief  ...
 *
 * \param p is the structure of the parameters for the operator 
 * \param m is a parameter of function
 * \param n  is a parameter of function
 * \param Pm1 is a parameter of function
 * \param Pm2 is a parameter of function
 * \param imamp1 is the first image data
 * \param imamp2 is the second image data
 */
void GrowRegion(Local_fr_param p, int m, int n, double *Pm1, double *Pm2, EFIDIRImage imamp1, EFIDIRImage imamp2);

/**
 * \ingroup local_fr
 * \fn void ReviseRegion(Local_fr_param p, double ma1, double ma2, EFIDIRImage imamp1, EFIDIRImage imamp2);
 * \brief intermediate calculation
 *
 * \param p is the structure of the parameters for the operator 
 * \param ma1 is a parameter of function
 * \param ma2 is a parameter of function
 * \param imamp1 is the first image data
 * \param imamp2 is the second image data
 *
 */
void ReviseRegion(Local_fr_param p, double ma1, double ma2, EFIDIRImage imamp1, EFIDIRImage imamp2);

/**
 * \ingroup local_fr
 * \fn int ComputeFreq_IDAN(Local_fr_param p, int i, int j, float *fx, float *fy, char *conf, EFIDIRImage imph, EFIDIRImage imfrest_LR);
 * \brief computing the local frequency value and its confidence index
 *
 * \param p is the structure of the parameters for the operator 
 * \param i,j is the index value 
 * \param fx is the real part of pixel value of output frequency image
 * \param fy is the imagery part of pixel value of output frequency image
 * \param conf is the pixel value of confidence image
 * \param imph is the phase image
 * \param imfrest_LR is the input frequency image
 */
/*FIXME int ComputeFreq_IDAN(Local_fr_param p, int i, int j, float *fx, float *fy, char *conf, EFIDIRImage imph, EFIDIRImage imfrest_LR);*/
void ComputeFreq_IDAN(Local_fr_param p, int i, int j, float *fx, float *fy, char *conf, EFIDIRImage imph, EFIDIRImage imfrest_LR);

/**
 * \ingroup local_fr
 * \fn void local_fr(Local_fr_param p, EFIDIRImage inputImageM, EFIDIRImage inputImageS, EFIDIRImage inputImageF, EFIDIRImage inputImageP, EFIDIRImage outputImageF, EFIDIRImage outputReg, EFIDIRImage outputConf );
 * \brief function of estimation of the local frequencies within an interferogram
 * Function allowing to estimate the local frequencies in an interferogram
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImageM is the float format master intensity image 
 * \param inputImageS is the float format slave intensity image 
 * \param inputImageF is the complex float format LR frequency image
 * \param inputImageP is the binary format uncompensated phase image
 * \param outputImageF is the image of frequency
 * \param outputReg is the image
 * \param outputConf is the binary format confidence image
 *
 */
void local_fr(Local_fr_param p, EFIDIRImage inputImageM, EFIDIRImage inputImageS, EFIDIRImage inputImageF, EFIDIRImage inputImageP, EFIDIRImage outputImageF,EFIDIRImage outputReg, EFIDIRImage outputConf );

#endif
