/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup interferometry
 * \defgroup interfero_filter Filter of interferogram in order to verify the result of phase unwrapping
 */
/**
 * \ingroup interfero_filter
 * \file iff_interfero_filter.h
 * \author Yajing Yan (LISTIC,LGIT)
 * 
 * \brief operator of fitering an interferogram
 * 
 * file includes (stucture, prototype de fonction) of the operator interfero_filter
 * 
 * 
 */

#ifndef IFF_INTERFERO_FILTER_H
#define IFF_INTERFERO_FILTER_H

#include <stdio.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "efidir_complex.h"
#include "efidir_string.h"
#include "com_coherence_phase.h"


/**
 * \ingroup interfero_filter
 * \typedef Interfero_filter_param
 *
 * \brief reference on interfero_filter_param structure
 *
 */
/**
 * \ingroup interfero_filter
 * \struct interfero_filter_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_interfero_filter_param{
  int size; /*!< size of filter window */
}interfero_filter_param, *Interfero_filter_param;

/**
 * \ingroup interfero_filter
 * \fn new_interfero_filter_param()
 *
 * \return A new reference on allocated interfero_filter_param structure
 *
 * \brief Create a new Interfero_filter_param reference
 *
 */
static Interfero_filter_param new_interfero_filter_param();
/**
 * \ingroup interfero_filter
 * \fn free_interfero_filter_param(Interfero_filter_param p)
 *
 * \param p A reference on a interfero_filter_param structure
 *
 * \brief Free an allocated Interfero_filter_param reference
 *
 */
void free_interfero_filter_param(Interfero_filter_param p);

/**
 * \ingroup interfero_filter
 * \fn void define_interfero_filter_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_interfero_filter_param(char *extra_description);

/**
 * \ingroup interfero_filter
 * \fn Interfero_filter_param get_interfero_filter_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Interfero_filter_param that contains the parameters.
 */
Interfero_filter_param get_interfero_filter_param();


/**
 * \ingroup interfero_filter
 * \fn void interfero_filter(Interfero_filter_param p, EFIDIRImage Image_fr, EFIDIRImage Image_int, EFIDIRImage Image_amp, EFIDIRImage outputImage);
 * \brief function of filtering the interferometric phase 
 * Function allowing to filter the interferogram with the local frequencies
 *
 * \param p is the structure of the parameters for the operator
 * \param Image_fr is the local frequencies image
 * \param Image_int is the complex interferogram
 * \param Image_amp is the amplitude image (cpx: re=master, im=slave)
 * \param outputImage is the filtered image
 *
 */

void interfero_filter(Interfero_filter_param p, EFIDIRImage Image_fr, EFIDIRImage Image_int, EFIDIRImage Image_amp, EFIDIRImage outputImage);
void interfero_filter_v2 (Interfero_filter_param p, EFIDIRImage amplitudecomplex, EFIDIRImage interferocomplex,
		EFIDIRImage phasemodel, EFIDIRImage outputImage,int xstart, int ystart);

void interfero_filter_v3(Interfero_filter_param p, EFIDIRImage amplitudecomplex, EFIDIRImage interferocomplex, EFIDIRImage phasemodel, EFIDIRImage freq_local, EFIDIRImage outputImage, int xstart, int ystart);

void rescale_coherence_phase(Coherence_phase_param p, EFIDIRImage input_phase_image,
		EFIDIRImage input_coherence_image, EFIDIRImage input_amplitude_master, EFIDIRImage input_amplitude_slave,
		EFIDIRImage output_amplitude_master, EFIDIRImage output_amplitude_slave, EFIDIRImage output_phase_image, EFIDIRImage output_coherence_image,
		int amp_choice);

void build_interferogram_rescaled(EFIDIRImage master, EFIDIRImage slave, Coherence_phase_param coh_phi_param, EFIDIRImage interferogram);

#endif
