/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup 3D_inversion
 * \defgroup fuzzy_projection Fuzzy projection of 3D displacement in LOS / azimuth direction
 */
/**
 * \ingroup fuzzy_projection
 * \file iff_fuzzy_projection.h
 * \author Yajing Yan 
 * 
 * \brief operator of processing the projection of 3D displacement in LOS / azimuth direction by fuzzy arithmetic
 * 
 * file includes (stucture, prototype de fonction) of the operator fuzzy_projection
 * 
 * 
 */

#ifndef IFF_FUZZY_PROJECTION_H
#define IFF_FUZZY_PROJECTION_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "efidir_string.h"
#include "iff_fuzzy_arth.h"

/**
 * \ingroup fuzzy_projection
 * \typedef Fuzzy_projection_param
 *
 * \brief reference on fuzzy_projection_param structure
 *
 */
/**
 * \ingroup fuzzy_projection
 * \struct fuzzy_projection_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_fuzzy_projection_param
{
  int nothing;
}fuzzy_projection_param, *Fuzzy_projection_param;

/**
 * \ingroup fuzzy_projection
 * \fn new_fuzzy_projection_param()
 *
 * \return A new reference on allocated fuzzy_projection_param structure
 *
 * \brief Create a new Fuzzy_projection_param reference 
 *
 */
static Fuzzy_projection_param new_fuzzy_projection_param();
/**
 * \ingroup fuzzy_projection
 * \fn free_fuzzy_projection_param(Fuzzy_projection_param p)
 *
 * \param p A reference on a fuzzy_projection_param structure
 *
 * \brief Free an allocated Fuzzy_projection_param reference 
 *
 */
void free_fuzzy_projection_param(Fuzzy_projection_param p);


/**
 * \ingroup fuzzy_projection
 * \fn void define_fuzzy_projection_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_fuzzy_projection_param(char *extra_description);

/**
 * \ingroup fuzzy_projection
 * \fn Fuzzy_projection_param get_fuzzy_projection_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Fuzzy_projection_param that contains the parameters.
 */
Fuzzy_projection_param get_fuzzy_projection_param();

/**
 * \ingroup fuzzy_projection
 * \fn void fuzzy_projection(Fuzzy_projection_param p, EFIDIRImage Component_E, EFIDIRImage Component_N, EFIDIRImage Component_Up, EFIDIRImage Sigma_E, EFIDIRImage Sigma_N, EFIDIRImage Sigma_Up, double *projection, EFIDIRImage outputImage, EFIDIRImage outputSigma);
 * \brief function of fuzzy projection
 * Function allowing to calculate the projection of 3D displacement into LOS / azimuth direction by fuzzy arithmetic
 *
 * \param p is the structure of the parameters for the operator 
 * \param Component_E is the E component image
 * \param Component_N is the N component image
 * \param Component_Up is the Up component image
 * \param Sigma_E is the uncertainty image of E component
 * \param Sigma_N is the uncertainty image of N component
 * \param Sigma_Up is the uncertainty image of Up component
 * \param projection is the vector of projection
 * \param outputImage is the projection image
 * \param outputSigma is the associated sigma image
 * 
 */
void fuzzy_projection(Fuzzy_projection_param p, EFIDIRImage Component_E, EFIDIRImage Component_N, EFIDIRImage Component_Up, EFIDIRImage Sigma_E, EFIDIRImage Sigma_N, EFIDIRImage Sigma_Up, double *projection, EFIDIRImage outputImage, EFIDIRImage outputSigma);

#endif
