/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup 3D_inversion
 * \defgroup fuzzy_inversion Fuzzy inversion 
 */
/**
 * \ingroup fuzzy_inversion
 * \file iff_fuzzy_inversion.h
 * \author Yajing Yan 
 * 
 * \brief operator of calculating the surface displacement in a (E,N,U) framework from the displacements of different projections with fuzzy arithmetic
 * 
 * file includes (stucture, prototype de fonction) of the operator fuzzy_inversion
 * 
 * 
 */

#ifndef IFF_FUZZY_INVERSION_H
#define IFF_FUZZY_INVERSION_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_matrix_op.h"
#include "efidir_string.h"
#include "iff_fuzzy_arth.h"

/**
 * \ingroup fuzzy_inversion
 * \typedef Fuzzy_inversion_param
 *
 * \brief reference on fuzzy_inversion_param structure
 *
 */
/**
 * \ingroup fuzzy_inversion
 * \struct fuzzy_inversion_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_fuzzy_inversion_param
{
  int process; /*!< processing index */
  float X_min; /*!< first coordonate in X to process */
  float Y_min; /*!< first coordonate in Y to process */
  float X_max; /*!< last coordonate in X to process */
  float Y_max; /*!< last  coordonate in Y to process */
  int n; /*!< number of projections to analyze */
  int index; /*!< output file type index */
}fuzzy_inversion_param, *Fuzzy_inversion_param;

/**
 * \ingroup fuzzy_inversion
 * \fn new_fuzzy_inversion_param()
 *
 * \return A new reference on allocated fuzzy_inversion_param structure
 *
 * \brief Create a new Fuzzy_inversion_param reference 
 *
 */
static Fuzzy_inversion_param new_fuzzy_inversion_param();
/**
 * \ingroup fuzzy_inversion
 * \fn free_fuzzy_inversion_param(Fuzzy_inversion_param p)
 *
 * \param p A reference on a fuzzy_inversion_param structure
 *
 * \brief Free an allocated Fuzzy_inversion_param reference 
 *
 */
void free_fuzzy_inversion_param(Fuzzy_inversion_param p);


/**
 * \ingroup fuzzy_inversion
 * \fn void define_fuzzy_inversion_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_fuzzy_inversion_param(char *extra_description);

/**
 * \ingroup fuzzy_inversion
 * \fn Fuzzy_inversion_param get_fuzzy_inversion_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Fuzzy_inversion_param that contains the parameters.
 */
Fuzzy_inversion_param get_fuzzy_inversion_param();

/**
 * \ingroup fuzzy_inversion
 * \fn void fuzzy_inversion(Fuzzy_inversion_param p, EFIDIRImage *def, EFIDIRImage *mask, EFIDIRImage *error, int nb, EFIDIRImage *output_series);
 * \brief function of fuzzy inversion
 * Function allowing to calculate the fuzzy distribution and matrix for fuzzy inversion
 *
 * \param p is the structure of the parameters for the operator 
 * \param def is the different projections of deformation
 * \param mask is the masks of different projections of deformation
 * \param error is the uncertainty image of different projections
 * \param nb is the number of projection
 * \param output_series is the output image series (displacement, sigma)
 * 
 */
void fuzzy_inversion(Fuzzy_inversion_param p, EFIDIRImage *def, EFIDIRImage *mask, EFIDIRImage *error,  int nb, EFIDIRImage *output_series);

#endif
