/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup infos_fusion
 * \defgroup fusion_scale Fusion of different scales of local frequencies
 */
/**
 * \ingroup fusion_scale
 * \file iff_fusion_scale.h
 * \author Yajing Yan
 * 
 * \brief operator of different scales fusion
 * 
 * file includes (stucture, prototype de fonction) of the operator fusion_scale
 * 
 * 
 */

#ifndef IFF_FUSION_SCALE_H
#define IFF_FUSION_SCALE_H

#include <stdio.h>
#include "efidir_image.h"
#include "efidir_param.h"


/**
 * \ingroup fusion_scale
 * \typedef Fusion_scale_param
 *
 * \brief reference on fusion_scale_param structure
 *
 */
/**
 * \ingroup fusion_scale
 * \struct fusion_scale_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_fusion_scale_param{
  int scale; /*!< scale of image to output */
  int Ystart; /*!< first line of interferogram with relative to intensity image */
}fusion_scale_param, *Fusion_scale_param;

/**
 * \ingroup fusion_scale
 * \fn new_fusion_scale_param()
 *
 * \return A new reference on allocated fusion_scale_param structure
 *
 * \brief Create a new Fusion_scale_param reference 
 *
 */
static Fusion_scale_param new_fusion_scale_param();
/**
 * \ingroup fusion_scale
 * \fn free_fusion_scale_param(Fusion_scale_param p)
 *
 * \param p A reference on a fusion_scale_param structure
 *
 * \brief Free an allocated Fusion_scale_param reference 
 *
 */
void free_fusion_scale_param(Fusion_scale_param p);


/**
 * \ingroup fusion_scale
 * \fn void define_fusion_scale_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_fusion_scale_param(char *extra_description);

/**
 * \ingroup fusion_scale
 * \fn Fusion_scale_param get_fusion_scale_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Fusion_scale_param that contains the parameters.
 */
Fusion_scale_param get_fusion_scale_param();

/**
 * \ingroup fusion_scale
 * \fn float combinaison(float m, float n);
 * \brief function of fusion of two measurement
 * Function allowing to combine two measurements
 *
 * \param m is the first measurment to combine
 * \param n is the second measurement to combine
 * \return the result of combinaision
 *
 */
 float combinaison(float m, float n);


/**
 * \ingroup fusion_scale
 * \fn void fusion_scale(Fusion_scale_param p, EFIDIRImage *frequency_series, EFIDIRImage *inconf_series, EFIDIRImage *cohe_series, EFIDIRImage scale_image, EFIDIRImage frequency_image, EFIDIRImage conf_image);
 * \brief function of fusion of different scale images
 * Function allowing to combine different scale images
 *
 * \param p is the structure of the parameters for the operator 
 * \param frequency_series is a series of images of different scale
 * \param inconf_series is the confidence image associated with frequency images series
 * \param cohe_series is the coherence images series
 * \param scale_image is the scale information image
 * \param frequency_image is the final fusion image 
 * \param conf_image is the confidence image of the final fusion image
 *
 */
void fusion_scale(Fusion_scale_param p, EFIDIRImage *frequency_series, EFIDIRImage *inconf_series, EFIDIRImage *cohe_series, EFIDIRImage scale_image, EFIDIRImage frequency_image, EFIDIRImage conf_image);


#endif
