/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup infos_fusion
 * \defgroup fusion Illustration of fusion of different Tracks
 */
/**
 * \ingroup fusion
 * \file iff_fusion.h
 * \author Yajing Yan
 * 
 * \brief operator of generation of an image for illustration of fusion
 * 
 * file includes (stucture, prototype de fonction) of the operator fusion
 * 
 * 
 */

#ifndef IFF_FUSION_H
#define IFF_FUSION_H

#include <stdio.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "efidir_string.h"


/**
 * \ingroup fusion
 * \typedef Fusion_param
 *
 * \brief reference on fusion_param structure
 *
 */
/**
 * \ingroup fusion
 * \struct fusion_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_fusion_param{
  int type;/*!< type of fusion */
  int nb; /*!< number of Tracks */
  double lon_start; /*!< minimum value of longitude*/
  double lon_last; /*!< maximum value of longtitude */
  double lat_start; /*!< minimum value of latitude*/
  double lat_last; /*!< maximum value of latitude */
  int index; /*!< 0: without fault trace, 1: with fault trace */
  float shiftX; /*!< shift in column of fault position */
  float shiftY; /*!< shift in line of fault position */
  float t1; /*!< threshold value of correlation error */
  float t2; /*!< threshold value of DInSAR error */
}fusion_param, *Fusion_param;

/**
 * \ingroup fusion
 * \fn new_fusion_param()
 *
 * \return A new reference on allocated fusion_param structure
 *
 * \brief Create a new Fusion_param reference
 *
 */
static Fusion_param new_fusion_param();
/**
 * \ingroup fusion
 * \fn free_fusion_param(Fusion_param p)
 *
 * \param p A reference on a fusion_param structure
 *
 * \brief Free an allocated Fusion_param reference
 *
 */
void free_fusion_param(Fusion_param p);

/**
 * \ingroup fusion
 * \fn void define_fusion_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_fusion_param(char *extra_description);

/**
 * \ingroup fusion
 * \fn Fusion_param get_fusion_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Fusion_param that contains the parameters.
 */
Fusion_param get_fusion_param();

/**
 * \ingroup fusion
 * \fn void fusion_datatype(Fusion_param p, EFIDIRImage *def_series, EFIDIRImage *mask_series, EFIDIRImage *error_series, char *fault_trace_file, EFIDIRImage outputImage);
 * \brief function of generation of a fusionstration image
 * Function allowing to generate an image for illustration of fusion
 *
 * \param p is the structure of the parameters for the operator
 * \param def_series is the deformation image series of different Tracks
 * \param mask_series is the mask series
 * \param error_series is the sigma series
 * \param fault_trace_file is the file containing the coordonate of the fault
 * \param outputImage is the image for illustration of fusion de different Tracks
 *
 */
void fusion_datatype(Fusion_param p, EFIDIRImage *def_series, EFIDIRImage *mask_series, EFIDIRImage *error_series, char *fault_trace_file, EFIDIRImage outputImage);

/**
 * \ingroup fusion
 * \fn void fusion_datanumber(Fusion_param p, EFIDIRImage *def_series, EFIDIRImage *mask_series, EFIDIRImage *error_series, char *fault_trace_file, EFIDIRImage outputImage);
 * \brief function of generation of a fusionstration image
 * Function allowing to generate an image for illustration of fusion (number of available data)
 *
 * \param p is the structure of the parameters for the operator
 * \param def_series is the deformation image series of different Tracks
 * \param mask_series is the mask series
 * \param error_series is the sigma series
 * \param fault_trace_file is the file containing the coordonate of the fault
 * \param outputImage is the image for illustration of fusion de different Tracks
 *
 */
void fusion_datanumber(Fusion_param p, EFIDIRImage *def_series, EFIDIRImage *mask_series,  EFIDIRImage *error_series, char *fault_trace_file, EFIDIRImage outputImage);


#endif
