/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup interferometry
 * \defgroup filter_multiscale Multiscale interferometric filter
 */
/**
 * \ingroup filter_multiscale
 * \file iff_filter_multiscale.h
 * \author Yajing Yan (LISTIC)
 * 
 * \brief operator of fitering an interferogram by local frequencies of multiscales
 * 
 * file includes (stucture, prototype de fonction) of the operator filter_multiscale
 * 
 * 
 */

#ifndef IFF_FILTER_MULTISCALE_H
#define IFF_FILTER_MULTISCALE_H

#include <stdio.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "efidir_complex.h"
#include "efidir_string.h"


/**
 * \ingroup filter_multiscale
 * \typedef Filter_multiscale_param
 *
 * \brief reference on filter_multiscale_param structure
 *
 */
/**
 * \ingroup filter_multiscale
 * \struct filter_multiscale_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_filter_multiscale_param{
  int size; /*!< size of filter window */
  int scale; /*!< scale of image to output */
  int shiftx; /*!< shift of columns of scale image with respect to image to output */
  int shifty; /*!< shift of lines of scale image with respect to image to output */
  int index; /*!< output data type index */
}filter_multiscale_param, *Filter_multiscale_param;

/**
 * \ingroup filter_multiscale
 * \fn new_filter_multiscale_param()
 *
 * \return A new reference on allocated filter_multiscale_param structure
 *
 * \brief Create a new Filter_multiscale_param reference
 *
 */
static Filter_multiscale_param new_filter_multiscale_param();
/**
 * \ingroup filter_multiscale
 * \fn free_filter_multiscale_param(Filter_multiscale_param p)
 *
 * \param p A reference on a filter_multiscale_param structure
 *
 * \brief Free an allocated Filter_multiscale_param reference
 *
 */
void free_filter_multiscale_param(Filter_multiscale_param p);

/**
 * \ingroup filter_multiscale
 * \fn void define_filter_multiscale_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_filter_multiscale_param(char *extra_description);

/**
 * \ingroup filter_multiscale
 * \fn Filter_multiscale_param get_filter_multiscale_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Filter_multiscale_param that contains the parameters.
 */
Filter_multiscale_param get_filter_multiscale_param();


/**
 * \ingroup filter_multiscale
 * \fn void filter_multiscale(Filter_multiscale_param p, EFIDIRImage *Fr_series, int number_of_series, EFIDIRImage Interferogram, EFIDIRImage Amplitude, EFIDIRImage scale_image, EFIDIRImage outputImage);
 * \brief function of filtering the unwrapped interferometric phase 
 * Function allowing to filter the interferograms with the local frequencies of multiscale
 *
 * \param p is the structure of the parameters for the operator
 * \param Fr_series is the local frequencies image series
 * \param number_of_series is the number of images in the series
 * \param Interferogram is the interferogram at the same scale of image to output
 * \param Amplitude is the amplitude image at the same scale of image to output (re: master, im: slave)
 * \param scale_image is the scale image
 * \param outputImage is the filtered image
 *
 */

void filter_multiscale(Filter_multiscale_param p, EFIDIRImage *Fr_series, int number_of_series, EFIDIRImage Interferogram, EFIDIRImage Amplitude, EFIDIRImage scale_image, EFIDIRImage outputImage);

#endif
