/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
* \ingroup divers
* \defgroup fault_trace Draw the trace of a fault in a binary image
*/

/**
* \ingroup fault_trace
* \file iff_fault_trace.h
* \brief operator of drawing the trace of a fault into a binary image
* \author Yajing Yan
*
*/


#ifndef IFF_FAULT_TRACE_H
#define IFF_FAULT_TRACE_H

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#include<string.h>
#include"efidir_image.h"
#include"efidir_param.h"
#include"efidir_assert.h"

/**
* \ingroup fault_trace
* \typedef typedef Fault_trace_param
*
* \brief structure that contains the parameters of the operator
*
*/
typedef struct struct_fault_trace_param
{
  double max_lat;/*latitude value of the first pixel*/
  double min_lon;/*longitude value of the first pixel*/
  double min_lat;/*latitude value of the last pixel*/
  double max_lon;/*longitude value of the last pixel*/
  int r;/*number of lines of the image*/
  int c;/*number of colums of the image*/
  int w;/*method of simulating of the fault*/
  int inval;/*interval of point for the simulation*/
}fault_trace_param,*Fault_trace_param;

/**
 * \ingroup fault_trace
 * \fn Fault_trace_param new_fault_trace_param();
 *
 * \return A new reference on allocated fault_trace_param structure
 *
 * \brief Create a new Fault_trace_param reference
 *
 */

static Fault_trace_param new_fault_trace_param();

/**
 * \ingroup fault_trace
 * \fn void free_fault_trace_param(Fault_trace_param p);
 *
 * \param p A reference on a fault_trace_param structure
 *
 * \brief Free an allocated Fault_trace_param reference
 *
 */

void free_fault_trace_param(Fault_trace_param p);

/**
* \ingroup fault_trace
* \fn void define_fault_trace_param(char *extra_description);
* \brief Default edifir function that defines parameters
*
* \param extra_description Extra description for multi definition of the same parameter(NULL is accepted)
*
*/
 void define_fault_trace_param(char *extra_description);

/**
* \ingroup fault_trace
* \fn Fault_trace_param get_fault_trace_param();
* \brief Default edifir function that gets parameters
*
* \return A fault_trace_param that contains the parameters
*
*/
Fault_trace_param get_fault_trace_param();

/**
* \ingroup fault_trace
* \fn void transpose(float **A,int line, int column, float **At);
* \brief function of transpose of a matrix
* function calculating the transpose of a matrix
*
* \param A is the matrix whose transpose we are looking for 
* \param line is the number of rows of matrix A
* \param column is the number of columns of matrix A
* \param At is the transpose of matrix A
*
*/
void transpose(float **A,int line, int column, float **At);


/**
* \ingroup fault_trace
* \fn void inversion(float **a,float *b,float *x,int m);
* \brief function of the resolution of a linear equation
* function looking for the root of a linear equation like ax=b
*
* \param a is a 2 dimension matrix that representes the left part of the equation
* \param b is a 1 dimension matrix that representes the right part of the equation
* \param x is a 1 dimension matrix that representes the root of the equation
* \param m is the size of the matrix
*
*/
void inversion(float **a,float *b,float *x,int m);

/**
 * \ingroup fault_trace
 * \fn void fault_trace(Fault_trace_param p, char *file_name_lat, char *file_name_lon,EFIDIRImage outputImage);
 * \brief function that displays an image from a ASCII file
 *
 * \param p operator parameters
 * \param file_name_lat is the name of the ASCII file containing the values of latitudes
 * \param file_name_lon is the name of the ASCII file containing the values of longitudes
 * \param outputImage is the image to display
 *
 */
void fault_trace(Fault_trace_param p,char *file_name_lat,char *file_name_lon,EFIDIRImage outputImage);

#endif
