/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
* \ingroup interferometry
* \defgroup correction_atm Substract the atmospheric fringes from interferogram
*/

/**
* \ingroup correction_atm
* \file iff_correction_atm.h
* \author Yajing Yan
*
* \brief operator of correction of atmospheric fringes
*
* file includes structures, prototypes of function of the operator correction_atm
*
*/

#ifndef IFF_CORRECTION_ATM_H
#define IFF_CORRECTION_ATM_H

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#include<string.h>
#include"efidir_image.h"
#include"efidir_param.h"
#include"efidir_matrix_op.h"

/**
 * \ingroup correction_atm
 * \typedef Correction_atm_param
 *
 * \brief reference on correction_atm_param structure
 *
 */

/**
 * \ingroup correction_atm
 * \struct correction_atm_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_correction_atm_param{
  int xstart; /*!< first column of image to process */
  int ystart; /*!< first line of image to process */
  int nbx; /*!< number of columns of image to process */
  int nby; /*!< number of lines of image to process */
  int dx; /*!< offset in x of DEM with respect to image */
  int dy; /*!< offset in y of DEM with respect to image */
  int index; /*!< processing index */
  float coef; /*!< coefficient phase/topo */
}correction_atm_param, *Correction_atm_param;

/**
 * \ingroup correction_atm
 * \fn new_correction_atm_param()
 *
 * \return A new reference on allocated correction_atm_param structure
 *
 * \brief Create a new Correction_atm_param reference 
 *
 */
static Correction_atm_param new_correction_atm_param();

/**
 * \ingroup correction_atm
 * \fn free_correction_atm_param(Correction_atm_param p)
 *
 * \param p A reference on a correction_atm_param structure
 *
 * \brief Free an allocated Correction_atm_param reference 
 *
 */
void free_correction_atm_param(Correction_atm_param p);


/**
 * \ingroup correction_atm
 * \fn void define_correction_atm_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_correction_atm_param(char *extra_description);

/**
 * \ingroup correction_atm
 * \fn Correction_atm_param get_correction_atm_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Correction_atm_param that contains the parameters.
 */
Correction_atm_param get_correction_atm_param();


/**
* \ingroup correction_atm
* \fn void atm_correction(Correction_atm_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage inputDEM, char *filename, EFIDIRImage outputImage);
* \brief function of correction of atmospheric fringes
* function subtracting atmospheric fringes from interferogram
*
* \param p is the structure of the parameters for the operator
* \param inputImage is the image to be corrected
* \param inputMask is the mask of image to be corrected
* \param inputDEM is the DEM 
* \param filename is the file containing the positions of windows
* \param outputImage is the image after the correction
*
*/

void atm_correction(Correction_atm_param p, EFIDIRImage inputImage,EFIDIRImage inputMask, EFIDIRImage inputDEM, char *filename, EFIDIRImage outputImage);
#endif

