/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup infos_fusion
 * \defgroup binary_image Formation of image from binary file
 */
/**
 * \ingroup binary_image
 * \file iff_binary_image.h
 * \author Yajing Yan (LISTIC,LGIT)
 * 
 * \brief operator of generation of images from a binary file
 * 
 * file includes (stucture, prototype de fonction) of the operator binary_image
 * 
 * 
 */

#ifndef IFF_BINARY_IMAGE_H
#define IFF_BINARY_IMAGE_H

#include <stdio.h>
#include "efidir_image.h"
#include "efidir_param.h"

#include "efidir_string.h"


/**
 * \ingroup binary_image
 * \typedef Binary_image_param
 *
 * \brief reference on binary_image_param structure
 *
 */
/**
 * \ingroup binary_image
 * \struct binary_image_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_binary_image_param{
  float lon_start; /*!< minimum value of longitude*/
  float lon_last; /*!< maximum value of longtitude */
  float lat_start; /*!< minimum value of latitude*/
  float lat_last; /*!< maximum value of latitude */
  float step_lon; /*!< step in X */
  float step_lat; /*!< step in Y */
}binary_image_param, *Binary_image_param;

/**
 * \ingroup binary_image
 * \fn new_binary_image_param()
 *
 * \return A new reference on allocated binary_image_param structure
 *
 * \brief Create a new Binary_image_param reference
 *
 */
static Binary_image_param new_binary_image_param();
/**
 * \ingroup binary_image
 * \fn free_binary_image_param(Binary_image_param p)
 *
 * \param p A reference on a binary_image_param structure
 *
 * \brief Free an allocated Binary_image_param reference
 *
 */
void free_binary_image_param(Binary_image_param p);

/**
 * \ingroup binary_image
 * \fn void define_binary_image_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_binary_image_param(char *extra_description);

/**
 * \ingroup binary_image
 * \fn Binary_image_param get_binary_image_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Binary_image_param that contains the parameters.
 */
Binary_image_param get_binary_image_param();

/**
 * \ingroup binary_image
 * \fn void binary_image(Binary_image_param p, char *file, EFIDIRImage *output_series);
 * \brief function of generation of a series of images from a binary file
 * Function allowing to generate a series of images for a binary file, program associated with the fuzzy inversion
 *
 * \param p is the structure of the parameters for the operator
 * \param file is the binary file
 * \param output_series is the image series constructed from the binary file
 *
 */
void binary_image(Binary_image_param p, char *file, EFIDIRImage *output_series);


/**
 * \ingroup binary_image
 * \fn void binary_image2(Binary_image_param p, char *file, EFIDIRImage *output_series);
 * \brief function of generation of an image from a binary file
 * Function allowing to generate a displacement image and its associated sigma image for a binary file, program associated with the fuzzy projection
 *
 * \param p is the structure of the parameters for the operator
 * \param file is the binary file
 * \param output_series is the displacement image and the associated sigma image
 *
 */
void binary_image2(Binary_image_param p, char *file, EFIDIRImage *output_series);




#endif
