/*    
 *    Copyright (c) 2015. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup goodies
 * \defgroup statevector statevector
 * \file efidir_statevector.h
 * \author Matthieu Volat (ISTerre)
 * 
 * \brief Load and interpolate state vectors
 */

#ifndef __EFIDIR_STATEVECTORS_H__
#define __EFIDIR_STATEVECTORS_H__

/**
 * \ingroup statevector
 * \brief State vector structure
 */
typedef struct state_vector { 
	double t; /**< Time in seconds since midnight */
	double x; /**< X position */
    double y; /**< Y position */
	double z; /**< Z position */
	double vx; /**< X velocity */
	double vy; /**< Y velocity */
	double vz; /**< Z velocity */
	struct state_vector *next; /**< Next vector in list (NULL if last) */
} *State_Vector;

/**
 * \ingroup statevector
 * \brief Interpolate a state vector from an array
 *
 * Returns a newly allocated state vector interpolated from the state vector
 * list to the specified time.
 * 
 * This function use linear interpolation to compute position and velocity.
 *
 * \param svlist   State vector list from which to interpolate
 * \param time     Time for the desired state vector, in seconds from 00:00:00
 *
 * \return Newly allocated, interpolated, state vector struct
 */
State_Vector svinterpolate(const State_Vector svlist, double time);

/**
 * \ingroup statevector
 * \brief Load a state vector file
 *
 * Read a state vector file into a State_Vector structure so functions
 * like svinterpolate() can be used.
 *
 * This state vector file use a simple structure: plaintext, one line per
 * state vector, compononents are given as a space-separated tuple
 * (t, x, y, z, vx, vy, vz), with no comments or headers. It is also
 * expected of the file to be sorted by time. This structure is definedto be
 * compatible with the state vectors files created and used by ROI_PAC.
 * 
 * \param filename  Path to the file to load
 *
 * \return Newly allocated state vector list
 */
State_Vector svloadfile(const char *filename);

/**
 * \ingroup statevector
 * \brief Free a state vector/state vector list
 *
 * Free a state vector list, possibly having only one element (like the ones
 * returned by svinterpolate()), from memory.
 *
 * \param svlist    State vector list to free
 */
void svfree(State_Vector svlist);

#endif
