/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup efidir 
 * \defgroup param_manager Parameters Auto-management
 * \file efidir_param.h
 * \author Flavien Vernier
 * \brief System of auto-management of parameters
 * 
 * This system manage automaicaly the parameters of the operator and executable.
 * It allows three execution modes:
 *  - command line mode: the volues of parameters are directly given on the command line
 *  - interactive mode: the application asks the values of the parameters
 *  - parameters file: the values of the parameters are read from a file 
 * 
 */
/**
 * \ingroup param_manager
 * \defgroup std_param Standard required parameters
 */
/**
 * \ingroup param_manager
 * \defgroup opt_param Optionnal parameters
 */
/**
 * \ingroup param_manager
 * \defgroup ser_param Parameters with series of values
 */

#ifndef EFIDIR_PARAM_H
#define EFIDIR_PARAM_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include <math.h>
#include <sys/times.h> 
#include <unistd.h>

#include "efidir_goodies.h"
#include "efidir_param_type.h"
#include "efidir_param_series.h"
#include "efidir_file.h"
#include "efidir_system.h"

static Param_manager params = NULL;
static int param_count=0;

/**
 * \ingroup std_param
 * \fn void define_io_param(char *identifier, Param_type type, Param_io io, bool series, bool optional, char *description, char *extra_description)
 * \brief Declaration of a new parameter
 *
 * \param identifier The name of the parameter.
 * \param type Type of the parameter
 * \param io Define the acces of param (IN/OUT)
 * \param series Set to true is the param require a series of values
 * \param optional Set to true is the param is optional
 * \param description Description of the parameter
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 *
 * The identifier of the parameter is used by the diferent modes to get the parameters.
 * To set a parameter in the explicit mode (command line) use "--identifier value".
 * In a parameter file mode, a line of a file is "identifier value"..
 * The description is used by the "--help" parameter to display a complet help on the executable and its parameters.
 * The description is also used by interactive mode to display the description of a parameter when it required.
 * A parameter define with this function is indispensable.
 *
 */
void define_io_param(char *identifier, Param_type type, Param_io io, bool series, bool optional, char *description, char *extra_description);

/**
 * \ingroup std_param
 * \fn void define_param(char *identifier, Param_type type, char *description, char *extra_description);
 * \brief Declaration of a new parameter
 *
 * \param identifier The name of the parameter.
 * \param type Type of the parameter
 * \param description Description of the parameter
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 *
 * The identifier of the parameter is used by the diferent modes to get the parameters.
 * To set a parameter in the explicit mode (command line) use "--identifier value".
 * In a parameter file mode, a line of a file is "identifier value"..
 * The description is used by the "--help" parameter to display a complet help on the executable and its parameters.
 * The description is also used by interactive mode to display the description of a parameter when it required.
 * A parameter define with this function is indispensable.
 *
 */
void define_param(char *identifier, Param_type type, char *description, char *extra_description);

/**
 * \ingroup opt_param
 * \fn void define_optional_param(char *identifier, Param_type type, char *description, char *extra_description);
 * \brief Declaration of a new optional parameter
 *
 * \param identifier The name of the parameter.
 * \param type Type of the parameter
 * \param description Description of the parameter
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 *
 * The identifier of the parameter is used by the diferent modes to get the parameters.
 * To set a parameter in the explicit mode (command line) use "--identifier value".
 * In a parameter file mode, a line of a file is "identifier value"..
 * The description is used by the "--help" parameter to display a complet help on the executable and its parameters.
 * The description is also used by interactive mode to display the description of a parameter when it required.
 * The differens with "define_param" function is that this parameter can be empty,
 * in the interactive mode it accepts the "null" value.
 *
 * The value are got as standard required parameter.
 *
 */
void define_optional_param(char *identifier, Param_type type, char *description, char *extra_description);

/**
 * \ingroup ser_param
 * \fn void define_series_param(char *identifier, Param_type type, char *description, char *extra_description);
 * \brief Declaration of a new parameter that waits a list of values
 *
 * \param identifier The name of the parameter.
 * \param type Type of the parameter values
 * \param description Description of the parameter
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 *
 * The identifier of the parameter is used by the diferent modes to get the parameters.
 * To set a parameter in the explicit mode (command line) use "--identifier values".
 * In a parameter file mode, a line of a file is "identifier values".
 *
 * The values, for numerical values can be define like "series[degin:end]", "series[degin:end:step]" or "series[val1, val2, val3...]"
 * for alphabetic values only like "series[val1, val2, val3...]".
 * The description is used by the "--help" parameter to display a complet help on the executable and its parameters.
 * The description is also used by interactive mode to display the description of a parameter when it required.
 * A parameter define with this function is indispensable.
 * 
 *
 */
void define_series_param(char *identifier, Param_type type, char *description, char *extra_description);

/**
 * \ingroup ser_param
 * \fn void define_optional_series_param(char *identifier, Param_type type, char *description, char *extra_description);
 * \brief Declaration of a new optional parameter that waits a list of values
 *
 * \param identifier The name of the parameter.
 * \param type Type of the parameter values
 * \param description Description of the parameter
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 *
 * The identifier of the parameter is used by the diferent modes to get the parameters.
 * To set a parameter in the explicit mode (command line) use "--identifier values".
 * In a parameter file mode, a line of a file is "identifier values".
 *
 * The values, for numerical values can be define like "series[degin:end]", "series[degin:end:step]" or "series[val1, val2, val3...]"
 * for alphabetic values only like "series[val1, val2, val3...]".
 * The description is used by the "--help" parameter to display a complet help on the executable and its parameters.
 * The description is also used by interactive mode to display the description of a parameter when it required.
 * A parameter define with this function is optional.
 * 
 *
 */
void define_optional_series_param(char *identifier, Param_type type, char *description, char *extra_description);


/**
 * \ingroup param_manager
 * \fn void init_param(int argc, char **argv);
 * \brief Run the parameter management .
 *
 */
void init_param(int argc, char **argv);

/**
 * \ingroup opt_param
 * \fn bool is_set_param(char *identifier);
 * \brief Check if a value is set to the given parameter.
 *
 * Check if a value is set to the given parameter. This can be useful with the optional parameters.
 * Be careful, if there is multi parameters with the same name, to use in same order as parameters definition.
 *
 * \return true if the parameter is set.
 */
bool is_set_param(char *identifier);

/**
 * \ingroup std_param
 * \fn int get_int_value_param(char *identifier);
 * \brief Return the value of the parameter as a integer value.
 *
 * \param identifier Parameter identifier.
 *
 * \return the required value. 
 */
int get_int_value_param(char *identifier);
/**
 * \ingroup std_param
 * \fn int get_long_value_param(char *identifier);
 * \brief Return the value of the parameter as a long integer value.
 *
 * This type of parameter must be used only if the value of the parameter can be greater than 2,147,483,647
 *
 * \param identifier Parameter identifier.
 *
 * \return the required value. 
 */
long get_long_value_param(char *identifier);
/**
 * \ingroup std_param
 * \fn double get_float_value_param(char *identifier);
 * \brief Return the value of the parameter as a float value.
 *
 * \param identifier Parameter identifier.
 *
 * \return the required value. 
 */
float get_float_value_param(char *identifier);
/**
 * \ingroup std_param
 * \fn double get_double_value_param(char *identifier);
 * \brief Return the value of the parameter as a double value.
 *
 * \param identifier Parameter identifier.
 *
 * \return the required value. 
 */
double get_double_value_param(char *identifier);
/**
 * \ingroup std_param
 * \fn char get_char_value_param(char *identifier);
 * \brief Return the value of the parameter as a character value.
 *
 * \param identifier Parameter identifier.
 *
 * \return the required value. 
 */
char get_char_value_param(char *identifier);
/**
 * \ingroup std_param
 * \fn char *get_string_value_param(char *identifier);
 * \brief Return the value of the parameter as a string value (char*).
 *
 * \param identifier Parameter identifier.
 *
 * \return the required value. 
 */
char *get_string_value_param(char *identifier);

/**
 * \ingroup ser_param
 * \fn int *get_int_series_values_param(char *identifier, int *nuber_of_values);
 * \brief Return the values of the parameter as an array of integer values.
 *
 * \param identifier Parameter identifier.
 * \param nuber_of_values Numbre of values of the series.
 *
 * \return an array of int. 
 */
int *get_int_series_values_param(char *identifier, int *nuber_of_values);
/**
 * \ingroup ser_param
 * \fn long *get_long_series_values_param(char *identifier, int *nuber_of_values);
 * \brief Return the values of the parameter as an array of long integer values.
 *
 * \param identifier Parameter identifier.
 * \param nuber_of_values Numbre of values of the series.
 *
 * \return an array of int. 
 */
long *get_long_series_values_param(char *identifier, int *nuber_of_values);
/**
 * \ingroup ser_param
 * \fn double *get_float_series_values_param(char *identifier, int *nuber_of_values);
 * \brief Return the values of the parameter as an array of double values.
 *
 * \param identifier Parameter identifier.
 * \param nuber_of_values Numbre of values of the series.
 *
 * \return an array of double. 
 */
double *get_float_series_values_param(char *identifier, int *nuber_of_values);
/**
 * \ingroup ser_param
 * \fn double *get_double_series_values_param(char *identifier, int *nuber_of_values);
 * \brief Return the values of the parameter as an array of double values.
 *
 * \param identifier Parameter identifier.
 * \param nuber_of_values Numbre of values of the series.
 *
 * \return an array of double. 
 */
double *get_double_series_values_param(char *identifier, int *nuber_of_values);
/**
 * \ingroup ser_param
 * \fn char *get_char_series_values_param(char *identifier, int *nuber_of_values);
 * \brief Return the values of the parameter as an array of character values.
 *
 * \param identifier Parameter identifier.
 * \param nuber_of_values Numbre of values of the series.
 *
 * \return an array of char. 
 */
char *get_char_series_values_param(char *identifier, int *nuber_of_values);
/**
 * \ingroup ser_param
 * \fn char **get_string_series_values_param(char *identifier, int *nuber_of_values);
 * \brief Return the values of the parameter as an array of string values (char*).
 *
 * \param identifier Parameter identifier.
 * \param nuber_of_values Numbre of values of the series.
 *
 * \return an array of char*. 
 */
char **get_string_series_values_param(char *identifier, int *nuber_of_values);

/**
 * \ingroup param_manager
 * \fn void save_param(char *file_name);
 * \brief Save the parameters and their values in a file defined by the given file_name.
 *
 * \param file_name Name of the save file.
 *
 */
void save_param(char *file_name);

/**
 * \ingroup param_manager
 * \fn void free_param();
 * \brief Stop and clean the parameter manager.
 *
 * Stop and clean the parameter manager.
 *
 */
void free_param();

/**
 * \ingroup param_manager
 * \fn void display_params();
 * \brief Display all parameters and their values.
 *
 * Display all parameters and their values.
 *
 */
void display_params();

/*
 * \ingroup param_manager
 * \fn void display_xml_params();
 * \brief Display all parameters in XML format.
 *
 * Display all parameters in XML format.
 *
 */
void display_xml_params();

/*  
 * \ingroup param_manager
 * \defgroup param_manager_advance advanced functions to manage the parameters
 */
/*
 * \ingroup param_manager_advance
 *
 * This set of fuction directly acces to parameters value and increases the level of the parameters.
 * Use of this functions requires a good knowlege of the parameter manager system.
 *
 */



/*
 * \ingroup param_manager_advance
 * \fn void set_int_value_param(char *identifier, int int_value);
 * \brief Set a value to a parameter.
 *
 * \param identifier Parameter identifier.
 * \param int_value Value of the parameter.
 *
 * This function can be used to directly set a int value to a parameter in the source code. 
 *
 */
void set_int_value_param(char *identifier, int int_value);

/*
 * \ingroup param_manager_advance
 * \fn void set_double_value_param(char *identifier, double double_value);
 * \brief Set a value to a parameter.
 *
 * \param identifier Parameter identifier.
 * \param double_value Value of the parameter.
 *
 * This function can be used to directly set a double value to a parameter in the source code. 
 *
 */
void set_double_value_param(char *identifier, double double_value);

/*
 * \ingroup param_manager_advance
 * \fn void set_char_value_param(char *identifier, char char_value);
 * \brief Set a value to a parameter.
 *
 * \param identifier Parameter identifier.
 * \param char_value Value of the parameter.
 *
 * This function can be used to directly set a char value to a parameter in the source code. 
 *
 */
void set_char_value_param(char *identifier, char char_value);

/*
 * \ingroup param_manager_advance
 * \fn void set_string_value_param(char *identifier, char* string_value);
 * \brief Set a value to a parameter.
 *
 * \param identifier Parameter identifier.
 * \param string_value Value of the parameter.
 *
 * This function can be used to directly set a string value to a parameter in the source code. 
 *
 */
void set_string_value_param(char *identifier, char* string_value);


#endif

