/* Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 * This file is part of EFIDIR tools.
 *
 * EFIDIR tool(s) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * EFIDIR tool(s) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "efidir_image.h"

#ifndef _EFIDIR_MATRIX_H
#define _EFIDIR_MATRIX_H

/**
 * \ingroup allocation
 * \fn void *alloc_vector(size_t size, int type);
 * \brief allocate memory for different type of vectors 
 *
 * \param size size allocated in term of number of element
 * \param type corresponds to ImageDataType
 *
 * \return void* the value returned is a pointer to the allocated memory.
 *
 */
#define alloc_vector(n, d)(__alloc_vector(n, d))
void *alloc_vector(size_t size, int type);

/**
 * \ingroup allocation
 * \def free_vector (void *v)
 * \brief free memory for different type of vectors
 *
 * \param v void* - an already allocated vector pointer
 *
 */
#define free_vector(v)(__efidir_free((void**)(&(v)), __FILE__, __LINE__))


/**
 * \ingroup allocation
 * \fn void **alloc_matrix(size_t nl, size_t nc, int dataType);
 * \brief allocate memory for different type of matrices 
 *
 * The matrix allocated with alloc_matrix is compatible with data of EFIDIRImage
 *
 * \param nl size allocated in term of number of lines
 * \param nc size allocated in term of number of element per line
 * \param dataType corresponds to ImageDataType 
 *
 * \return void** the value returned is a pointer to the allocated memory.
 *
 */
#define alloc_matrix(l, c, d)(__alloc_matrix(l, c, d))
void **alloc_matrix(size_t nl, size_t nc, int dataType);

/*
 * \ingroup allocation
 * \def free_matrix(m) (__free_matrix((void***)(&(m))))
 * \brief free memory for different type of matrices 
 *
 * free memory of matrix allocated with alloc_matrix
 *
 * \param m void** - an already allocated matrix pointer
 *
 */
/**
 * \ingroup allocation
 * \def free_matrix (void **m)
 * \brief free memory for different type of matrices 
 *
 * free memory of matrix allocated with alloc_matrix
 *
 * \param m void** - an already allocated matrix pointer
 *
 */
#define free_matrix(m)(__free_matrix((void***)(&(m))))
void __free_matrix(void ***m);

/**
 * \ingroup allocation
 * \fn void **alloc_matrix3D(size_t nl, size_t nc, size_t ns, int dataType);
 * \brief allocate memory for different type of matrices 
 *
 * The matrix allocated with alloc_matrix is compatible with data of EFIDIRImage
 *
 * \param ns size allocated in term of number of step
 * \param nl size allocated in term of number of lines
 * \param nc size allocated in term of number of element per line
 * \param dataType corresponds to ImageDataType 
 *
 * \return void** the value returned is a pointer to the allocated memory.
 *
 */
#define alloc_matrix3D(s, l, c, d)(__alloc_matrix3D(s, l, c, d))
void ***alloc_matrix3D(size_t ns, size_t nl, size_t nc, int dataType);

/*
 * \ingroup allocation
 * \def free_matrix3D(m) (__free_matrix((void***)(&(m))))
 * \brief free memory for different type of matrices 
 *
 * free memory of matrix allocated with alloc_matrix
 *
 * \param m void** - an already allocated matrix pointer
 *
 */
/**
 * \ingroup allocation
 * \def free_matrix3D (void **m)
 * \brief free memory for different type of matrices 
 *
 * free memory of 3D matrix allocated with alloc_matrix3D
 *
 * \param m void*** - an already allocated 3D matrix pointer
 *
 */
#define free_matrix3D(m)(__free_matrix3D((void****)(&(m))))
void __free_matrix3D(void**** m);

#endif
