/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup efidir_math_tools
 * \defgroup math useful math functions
 * \file efidir_math.h
 * \author Flavien Vernier and Yajing Yan
 *
 * \brief Useful math functions
 *
 */

#ifndef EFIDIR_MATH_H
#define EFIDIR_MATH_H

#include<math.h>

/**
 * \ingroup math
 * \def max(a,b)
 * \brief return the larger of the numerical values a and b
 *
 * \param a (number value: int, float, double or others...)
 * \param b (number value: int, float, double or others...)
 *
 * \return return the larger of the numerical values a and b
 */
#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/**
 * \ingroup math
 * \def min(a,b)
 * \brief return the smaller of the numerical values a and b
 *
 * \param a (number value: int, float, double or others...)
 * \param b (number value: int, float, double or others...)
 *
 * \return return the smaller of the numerical values a and b
 */
#ifndef min
#define min(a,b) (((a) < (b)) ? (a) : (b))
#endif

/**
 * \ingroup math
 * \def PI
 * \brief return PI value. PI corresponds to M_PI of math.h
 *
 * \return return PI double value
 */
#ifndef PI
#define PI M_PI
#endif


/**
 * \ingroup math
 * \def abs(a)
 * \brief return the absolute value of a number or a functional expression (value: int, float, double or others... )
 *
 * \param a the expression whose absolute value we need
 *
 * \return return the absolute value
 */
#ifndef abs
#define abs(a) (((a) >= 0) ? (a) : (-(a)))
#endif

/**
 * \ingroup math
 * \def swap(x,y)
 * \brief swap two numbers (value: int, float, double or others... )
 *
 * \param x is the first number to be swaped
 * \param y is the second number to be swaped
 * 
 */
#ifndef swap
#define swap(x,y) (t=(x),(x)=(y),(y)=t)
#endif


/**
 * \ingroup math
 * \def round(p)
 * \brief approximate a decimal number to its nearest integer
 *
 * \param p is a decimal number or a functional expression
 * \return an integer that is the frist approximation of the decimal number
 */
#ifndef round
#define round(p) (((p)>=0)?((int)floor((p)+0.5)):((int)ceil((p)-0.5)))
#endif

/**
 * \ingroup math
 * \def SQR(a)
 * \brief calculate a*a
 *
 * \param a is a number of type int, float, double......
 * \return its square value
 */
#ifndef SQR
#define SQR(a) ((a)*(a))
#endif


/**
 * \ingroup math
 * \def DEG2RAD(a)
 * \brief convert degree to radian
 *
 * \param a is a number that represent degree
 * \return the radian value
 */
#ifndef DEG2RAD
#define DEG2RAD(a) ((a)*M_PI/180.0)
#endif


/**
 * \ingroup math
 * \def RAD2DEG(a)
 * \brief convert radian to degree
 *
 * \param a is a number that represent radian
 * \return the degree value
 */
#ifndef RAD2DEG
#define RAD2DEG(a) ((a)*180.0/M_PI)
#endif

/**
 * \ingroup math
 * \def RAD2MET(a,lambda)
 * \brief convert radian to meter
 *
 * \param a is a number that represent radian
 * \param lambda is the wavelength
 * \return the meter value
 */
#ifndef RAD2MET
#define RAD2MET(a,lambda) ((a)*(lambda)/(4*PI))
#endif

/**
 * \ingroup math
 * \def MET2RAD(a,lambda)
 * \brief convert meter to radian
 *
 * \param a is a number that represent meter
 * \param lambda is the wavelength
 * \return the radian value
 */
#ifndef MET2RAD
#define MET2RAD(a,lambda) ((a)*4*PI/(lambda))
#endif


/**
 * \ingroup math
 * \def width_hh (a,b,c)
 * \brief compute the width of a parabola (y=ax2+bx+c) at half height
 *
 * \param a is the coefficient of order 2
 * \param b is the coefficient of order 1
 * \param c is the constant offset
 * \return the width at half height
 */
#ifndef width_hh
#define width_hh(a,b,c) ((SQR(b)-4*(a)*(c)>=0)?(sqrt((SQR(b)-4*(a)*(c))/2)/abs(a)): 0)
#endif

#endif
