/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __EFIDIR_IMAGE_LIST_H__
#define __EFIDIR_IMAGE_LIST_H__

#include "efidir_image.h"
#include "efidir_assert.h"
#include "efidir_allocation.h"
#include "efidir_param.h"
#include "efidir_file_txt.h"
#include "efidir_goodies.h"
#include "efidir_image_raster.h"
#include <string.h>
#include <math.h>
#include <stdlib.h>
//#include <ctype.h>
//#include <fcntl.h>
//#include <sys/stat.h>

#ifdef _WIN32
#include <io.h>
#endif

/**
 * \ingroup efidir 
 * \defgroup listimage_manager EFIDIR listimage manager
 * \file efidir_image_list.h
 * \author Sophie Martz (CNRS)
 * 
 * \brief EFIDIR listimage manager
 */


/**
 * \ingroup listimage_manager
 * \struct EFIDIRListImage
 * \brief EFIDIR List Image structure
 */
typedef struct EFIDIRListImage {
  int nbrImages;              /*!< Number of images */
  int nc;	              /*!< Number of image data columns of the shortest image in memory */
  int nr;                     /*!< Number of image data rows of the shortest image in memory */
  char *filename;             /*!< listImage file name */
  int samples;                /*!< Number of samples (columns) per image of the largest image */
  int lines;                  /*!< Number of lines per image of the largest image */
  int dataType;               /*!< Data type of all the images */
  int bands;                  /*!< Number of band of all the images */
  bool homogeneous;        /*!< True if all the images have the same number of lines and the same number of samples (column) */
  bool homogeneousSamples; /*!< True if all the images have the same number of samples (column) */
  EFIDIRImage *images;	      /*!< Table without any size to store EFIDIR images of the list */
  int *sampleOffsets;          /*!< Range offset relatively to the first image */
  int *lineOffsets;            /*!< Azimuth offset relatively to the first image */
} *EFIDIRListImage;

/*TODO read an image as list of image where each band is a image.*/

/*
 * \ingroup listimage_manager
 * \fn EFIDIRListImage alloc_listimage();
 * \brief Allocate an listimage empty structure
 *
 * \return address of the allocated listimage structure
 */
EFIDIRListImage alloc_listimage();

/*
 * \ingroup listimage_manager
 * \fn void free_listimage(EFIDIRListImage list);
 * \brief Free a previously allocated image structure
 *
 * \param list : address of the image structure
 */
void free_listimage(EFIDIRListImage list);

/**
 * \ingroup listimage_manager
 * \fn EFIDIRListImage read_listimage_open(char *filename);
 * \brief Open a list file for reading
 *
 * \param filename : the listimage file name
 * \return list : the new listimage structure
 */
EFIDIRListImage read_listimage_open(char *filename);

/**
 * \ingroup listimage_manager
 * \fn void read_listimage_data(EFIDIRListImage list);
 * \brief Read all images data from the listimage file
 *
 * \param list : address of the list structure
 */
void read_listimage_data(EFIDIRListImage list);

/**
 * \ingroup listimage_manager
 * \fn int read_listimage_line(EFIDIRListImage list, int l);
 * \brief Read the specified line of each image data from the list file
 *
 * \param list : address of the list structure
 * \param l : the line number
 * \return null if the specified line l is outside the image range
 */
int read_listimage_line(EFIDIRListImage list, int l);

/**
 * \ingroup listimage_manager
 * \fn int read_listimage_part(EFIDIRListImage list, int col, int row, int nCols, int nRows);
 * \brief Read the specified part of each image data from the list file
 *
 * Read all bands of the specified part of image data from the image file,
 * to switch band (see select_image_band).
 *
 * \param list : address of the list structure
 * \param col : the first col of the image part
 * \param row : the first row of the image part
 * \param nCols : number of columns
 * \param nRows : number of rows
 * \return null if the specified part is outside the image range
 */
int read_listimage_part(EFIDIRListImage list, int col, int row, int nCols, int nRows);


/**
 * \ingroup image_function
 * \fn void select_listimage_band(EFIDIRListImage list, int band);
 * \brief Change the current listimage band
 *
 * \param list : address of the list structure
 * \param band : the new current band number (from 0)
 */
void select_listimage_band(EFIDIRListImage list, int band);

/**
 * \ingroup listimage_manager
 * \fn void read_listimage_close(EFIDIRListImage list);
 * \brief Close the list images file
 *
 * \param list : address of the list structure
 */
void read_listimage_close(EFIDIRListImage list);


/**
 * \ingroup listimage_manager
 * \fn void alloc_listimage_data(EFIDIRListImage list, int nc, int nr);
 * \brief Allocate data of a list of images
 *
 * \param list address of the list image structure
 * \param nc the number of columns to allocate
 * \param nr the number of rows to allocate
 */
void alloc_listimage_data(EFIDIRListImage list, int nc, int nr);

/**
 * \ingroup listimage_manager
 * \fn EFIDIRListImage write_listimage_open(const char *name, const char *description, int samples, int lines, int bands, int dataType, int interleave, int nuberOfImages);
 * \brief Open an image file for writing
 * 
 * \param name the list image file name
 * \param description optionnal description of the images
 * \param samples number of samples
 * \param lines number of lines
 * \param bands number of bands (1 or more)
 * \param dataType type of data representation
 * \param interleave interleave method for multi-bands image
 * \param nuberOfImages nuber of images in the list
 * \return listimage the new list image structure
 */
EFIDIRListImage write_listimage_open(char *name, char *description, int samples, int lines, int bands, int dataType, int interleave, int nuberOfImages);

/**
 * \ingroup listimage_manager
 * \fn void write_listimage_data_all_bands(EFIDIRListImage list);
 * \brief Write the image data into the image files
 *
 * Write all bands of the image data into the image file.
 *
 * \param list address of the image structure
  *
 * \see select_image_band, 
 */
void write_listimage_data_all_bands(EFIDIRListImage list);

/**
 * \ingroup listimage_manager
 * \fn int write_listimage_line_all_bands(EFIDIRListImage list, int line);
 * \brief Write the specified line of image data into the image file
 *
 * Write all bands of the specified line of list image data into the image files,
 * to switch band (see select_image_band).
 *
 * \param list address of the image structure
 * \param line the line number
 * \return null if the specified line is outside the image range
 *
 * \see select_image_band, 
 */
int write_listimage_line_all_bands(EFIDIRListImage list, int line);

/**
 * \ingroup listimage_manager
 * \fn void write_listimage_close(EFIDIRListImage list);
 * \brief Close the list image file
 *
 * \param list address of the list image structure
 */
void write_listimage_close(EFIDIRListImage list);




/*--------------------------------------------------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------------------------------------------------*/





/**
 * \ingroup listimage_manager
 * \struct listimage_param
 *
 * \brief structure that contains the information to manage input and output list of imagr
 *
 */
typedef struct struct_listimage_param{
  char* input_file_name; /*!< file name of input list of images*/
  char* output_suffix; /*!< sufux to generate output list of images */
  char* output_dir; /*!< directory of output list of images */
  char* output_description; /*!< description of output images */
  int firstImage;              /*!< First image of the list to compute */
  int lastImage;               /*!< Last image of the list to compute */
}listimage_param, *Listimage_param;



/**
 * \ingroup listimage_manager
 * \fn Listimage_param new_listimage_param();
 * \brief Create a new Listimage_param reference 
 *
 * \return A new reference on allocated listimage_param structure
 *
 */
Listimage_param new_listimage_param();

/**
 * \ingroup listimage_manager
 * \fn void free_listimage_param(Listimage_param p);
 *
 * \brief Free an allocated Listimage_param reference 
 *
 * \param p A reference on a listimage_param structure
 *
 */
void free_listimage_param(Listimage_param p);

/**
 * \ingroup listimage_manager
 * \fn void define_listimage_param_with_inout_lists(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
 */
void define_listimage_param_with_inout_lists(char *extra_description);

/**
 * \ingroup listimage_manager
 * \fn Listimage_param get_listimage_param_with_inout_lists();
 * \brief Default efidir function that gets parameters
 *
 * \return A Listimage_param that contains the parameter values.
 *
 */
Listimage_param get_listimage_param_with_inout_lists();


/**
 * \ingroup listimage_manager
 * \fn char* build_output_file(char *input_file_name, char* output_suffix);
 * \brief Build the output file according an input file name and a suffix
 *
 * Build the output file according an input file name and a suffix.
 * The output file is placed in the same directory of the input file.
 * Each image of the output file is placed in the same directory of its corresponding input image if output_dir is NULL.
 * Each output name is the same of its input name plus the suffix.
 *
 * \param input_file_name base of name
 * \param output_suffix suffix to add at the end of input file name
 * \param output_dir directory of output files
 *
 * \return output file name.
 */
char* build_output_file(char *input_file_name, char* output_suffix, char* output_dir);

/**
 * \ingroup listimage_manager
 * \fn char* build_output_file_mix_by_2(char *input_file_name, char* output_suffix, char* output_dir, int mode);
 * \brief Build the output file according an input file name and a suffix
 *
 * Build the output file according an input file name and a suffix.
 * The output file is placed in the same directory of the input file.
 * Each image of the output file is placed in the same directory of its corresponding input image if output_dir is NULL.
 * Each output name is the mix of 2 input image name.
 *
 * \param input_file_name base of name
 * \param output_suffix suffix to add at the end of input file name
 * \param output_dir directory of output files
 * \param mode mix mode: 0 merge master image name with the others, 1 merge the image name with the next, 2 merge the image name with the second next...
 * \param master define the master image for mode 0 (not used for other mode)
 *
 * \return output file name.
 */
char* build_output_file_mix_by_2(Listimage_param p, int mode, int master);

#endif /*__EFIDIR_IMAGE_LIST_H__*/
