/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup image_manager 
 * \defgroup image_info info
 * \file efidir_image_info.h
 * \author Matthieu Volat (ISTerre), Flavien Vernier (LISTIC)
 * 
 * \brief EFIDIR image: manage info (geotype)
 */

#ifndef __EFIDIR_IMAGE_INFO_H__
#define __EFIDIR_IMAGE_INFO_H__


#include <float.h>
#include "efidir_image.h"

/**
 * \ingroup image_manager
 * \defgroup image_info image_info
 * \author Matthieu Volat (ISTerre)
 *
 * \brief Manage non-raster aspects of images
 */

/**
 * \ingroup image_info
 * \brief Type to store affine transform parameters
 *
 * See getgeotransform().
 */
typedef double Transform[6];

/**
 * \ingroup image_info
 * \brief Retrieve image geoaffine transformation parameters
 *
 * Images can be associated with a affine transform, consisting of six 
 * coefficients, which map pixel/line coordinates into georeferenced space 
 * using the following relationship:
 * 
 * \code{.c}
 *   xgeo = transform[0] + xpixel*transform[1] + ypixel*transform[2]
 *
 *   ygeo = transform[3] + xpixel*transform[4] + ypixel*transform[5]
 * \endcode
 *
 * Note that georeferencing is (obviously) not a given for an image,
 * getgeotransform() will return 0 if the image indeed have a valid
 * transform. In case of non-zero return, content of transform should be
 * discarded.
 *
 * Also note this usage is not a direct interface to ENVI's map info field,
 * but a more generalized API that most notably enable to use rotation.
 * Loading ENVI image georeferencing has no issue, but it is obviously not
 * recommended to have non-zero values for transform[2] and transform[4]. If
 * you find yourself with such images that you want to save without loosing
 * georeferencing, you must rotate their content and use setgeotransform()
 * to update the transform accordingly.
 *
 * \param img		Image to read the transform of
 * \param transform	Variable to store the transform in
 *
 * \return Zero (0) if returned transform is valid
 */
int getgeotransform(const EFIDIRImage img, Transform transform);

/**
 * \ingroup image_info
 * \brief Set or update image geoaffine transformation parameters
 *
 * Update an image georeferencing transform. See getgeotransform() to
 * see more on the transform parameter.
 *
 * If transform is NULL, the transform will be set to {0, 0, 0, 0, 0, 0} and
 * flagged as invalid.
 *
 * \param img		Image to set the transform of
 * \param transform	Variable to read the transform from
 *
 * \return Always return zero (success)
 */
int setgeotransform(EFIDIRImage img, const Transform transform);

#endif
