/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * \ingroup efidir 
 * \defgroup image_manager Image manager
 * \file efidir_image.h
 * \author Pascal Bellemain (GIPSA-lab), Flavien Vernier (LISTIC), Matthieu Volat (ISTerre)
 * 
 * \brief EFIDIR image manager
 */
/**
 * \ingroup image_manager 
 * \defgroup image_type types
 * \file efidir_image.h
 * \author Pascal Bellemain (GIPSA-lab), Flavien Vernier (LISTIC)
 * 
 * \brief EFIDIR image: type definitions
 */
/**
 * \ingroup image_manager 
 * \defgroup image_function functions
 * \file efidir_image.h
 * \author Pascal Bellemain (GIPSA-lab), Flavien Vernier (LISTIC)
 * 
 * \brief EFIDIR image: function definitions
 */
/**
 * \ingroup image_manager 
 * \defgroup image_bati BATI compatibility
 * \file efidir_image.h
 * \author Pascal Bellemain (GIPSA-lab), Flavien Vernier (LISTIC)
 */
/**
 * \addtogroup image_bati
 * 
 *  BATI types compatibility must be used only to integrated BATI code in EFIDIR.
 *
 *  Never use to develop in EFIDIR 
 */

#ifndef __EFIDIR_IMAGE_H__
#define __EFIDIR_IMAGE_H__

#include "efidir_deprecated.h"
#include "efidir_file.h"
#include "efidir_file_txt.h"
#include "efidir_string.h"
#include "efidir_complex.h"
#include "efidir_assert.h"
#include "efidir_allocation.h"
#include "efidir_boolean.h"
#include "efidir_statevector.h"

/**
 * \ingroup image_type
 * \enum ImageFileType
 * \brief suported input imgae type
 */
enum ImageFileType{
  FILE_TYPE_ENVI,
  FILE_TYPE_BATI,
  FILE_TYPE_RASTER,
  FILE_TYPE_ROIPAC,
  FILE_TYPE_CSK,
  FILE_TYPE_TSX,
  FILE_TYPE_IMAGEMAGICK,
  FILE_TYPE_OPENCV,
  FILE_TYPE_GDAL
};
static const char *ImageFileTypeName[] = {"ENVI", "BATI", "RASTER", "ROIPAC", "CSK", "T(S/D)X", "MAGICK", "OPENCV", "GDAL"};

/**
 * \ingroup image_type
 * \enum ImageDataType
 * \brief Image type of data representation constants.
 *
 * Constatn integer values of the image data types.
 * The order is define according to the ENVI header file format (.hdr).
 *
 */
enum ImageDataType {
	DATA_TYPE_UINT8       =  1, /*!< 8 bits unsigned byte data type */
	DATA_TYPE_INT16       =  2, /*!< 16 bits signed integer data type */
	DATA_TYPE_INT32       =  3, /*!< 32 bits signed long integer data type */
	DATA_TYPE_FLOAT32     =  4, /*!< 32 bits floating point data type */
	DATA_TYPE_FLOAT64     =  5, /*!< 64 bits double precision floating point data type */
	DATA_TYPE_CPLXFLOAT32 =  6, /*!< 2x32 bits single precision (float) complex data type */
	DATA_TYPE_CPLXFLOAT64 =  9, /*!< 2x64 bits double precision complex data type */
	DATA_TYPE_UINT16      = 12, /*!< 16 bits unsigned integer data type */
	DATA_TYPE_UINT32      = 13, /*!< 32 bits unsigned long integer data type */
	/* Extended datatypes */
	DATA_TYPE_CPLXINT16   = 33, /*!< 2x16 bits integer complex data type */
	DATA_TYPE_RGB         = 34, /*!< 24 bits color (red, green, blue) data type */
	DATA_TYPE_BGR         = 35, /*!< 24 bits color (blue, green, red) data type */
	DATA_TYPE_CPLXINT32   = 36, /*!< 2x32 bits integer complex data type */
};
static const char *ImageDataTypeName[] = {"Unknown", "UInt 8", "Int 16","Int 32","Float","Double","CPLX float 32", "Unknown", "Unknown", "CPLX float 64", "Unknown", "Unknown", "UInt 16", "UInt 32", "Int 64", "UInt 64", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Int 8", "CPLX int 16", "RGB", "BGR"};

/* deprecated
 * \ingroup image_type
 * \enum ImageDataTypeExtended
 * \brief Redefinition of image type of data representation constants.
 */
DEPRECATED(enum) ImageDataTypeExtended {
	DATA_TYPE_BYTE = DATA_TYPE_UINT8,
	DATA_TYPE_SHORT = DATA_TYPE_INT16,
	DATA_TYPE_INT = DATA_TYPE_INT32,
	DATA_TYPE_FLOAT = DATA_TYPE_FLOAT32,
	DATA_TYPE_DOUBLE = DATA_TYPE_FLOAT64,
	DATA_TYPE_COMPLEX8 = DATA_TYPE_CPLXFLOAT32,
	DATA_TYPE_COMPLEX16 = DATA_TYPE_CPLXFLOAT64,
	DATA_TYPE_WORD = DATA_TYPE_UINT16,
	DATA_TYPE_DWORD = DATA_TYPE_UINT32,
    /* Extended datatypes */
	DATA_TYPE_COMPLEX4 = DATA_TYPE_CPLXINT16,
};

/**
 * \ingroup image_type
 * \enum ImageInterleave
 * \brief Image data interleaving constants.
 */
enum ImageInterleave {
	INTERLEAVE_BSQ = 1, /*!< Band sequential interleaving */
	INTERLEAVE_BIP,     /*!< Band interleaved by pixel */
	INTERLEAVE_BIL      /*!< Band interleaved by line */
};
static const char *ImageInterleaveName[] = {"BSQ", "BIP", "BIL"};
/**
 * \ingroup image_type
 * \enum ImageByteOrder
 * \brief Image bytes order constants.
 */
enum ImageByteOrder {
	BYTE_ORDER_LSF, /*!< Least significant byte first data */
	BYTE_ORDER_MSF  /*!< Most significant byte first data (not supported) */
};
static const char *ImageByteOrderName[] = {"LSF", "MSF"};
/*
 * \ingroup image_manager
 * \typedef Str255
 * \brief Fixed length string.
 */
typedef char Str255[256];


/**
 * \ingroup image_type
 * \struct RGBColor
 * \brief color structure with red, green, blue components
 */
typedef struct RGBColor {
     unsigned char red;
     unsigned char green;
     unsigned char blue;
} RGBColor;
/**
 * \ingroup image_type
 * \struct BGRColor
 * \brief color structure with blue, green, red components
 */
typedef struct BGRColor {
     unsigned char blue;
     unsigned char green;
     unsigned char red;
} BGRColor;


/** 
 * /var Size of image data element according to the data type
 */
static int dataTypeSize[] = {
	0,
	sizeof(char),           // 1
	sizeof(short),          // 2
	sizeof(int),            // 3
	sizeof(float),          // 4
	sizeof(double),         // 5
	sizeof(Complex8),       // 6
	0,                      // 7
	0,                      // 8
	sizeof(Complex16),      // 9
	0,                      // 10
	0,                      // 11
	sizeof(unsigned short), // 12
	sizeof(unsigned int),   // 13
	sizeof(int64_t),        // 14
	sizeof(uint64_t),       // 15
	0,                      // 16
	0,                      // 17
	0,                      // 18
	0,                      // 19
	0,                      // 20
	0,                      // 21
	0,                      // 22
	0,                      // 23
	0,                      // 24
	0,                      // 25
	0,                      // 26
	0,                      // 27
	0,                      // 28
	0,                      // 29
	0,                      // 30
	0,                      // 31
	/* Extended datatypes */
	sizeof(char),           // 32
	sizeof(Complex4),       // 33
	sizeof(RGBColor),       // 34
	sizeof(BGRColor)        // 35
};

/**
 * \ingroup image_type
 * \struct ImageParam
 * \brief Image parameter structure
 */
typedef struct ImageParam_struct {
	char *name;              /*!> Parameter name */
	char *value;             /*!> Parameter string value */
	struct ImageParam_struct *next; /*!> Next parameter in the parameters list */
} *ImageParam;
/**
 * \ingroup image_type
 * \struct EFIDIRImage_struct
 * \brief EFIDIR Image structure
 */
typedef struct EFIDIRImage_struct {
	union {
	  unsigned char **u1;  /*!< 8 bits byte pointer type points on current band*/
	  char **s1;           /*!< 8 bits signed integer pointer type points on current band*/
	  short **s2;          /*!< 16 bits signed integer pointer type points on current band*/
	  int **s4;            /*!< 32 bits signed long integer pointer type points on current band*/
	  float **fl;          /*!< 32 bits floating point pointer type points on current band*/
	  double **db;         /*!< 64 bits double precision floating point pointer type points on current band*/
	  Complex4 **cx4;      /*!< 2x16 bits integer complex pointer type points on current band*/
	  Complex8 **cx8;      /*!< 2x32 bits single precision (float) complex pointer type points on current band*/
	  Complex16 **cx16;    /*!< 2x64 bits double precision complex pointer type points on current band*/
	  unsigned short **u2; /*!< 16 bits unsigned integer pointer type points on current band*/
	  unsigned int **u4;   /*!< 32 bits unsigned long integer pointer type points on current band*/
      int64_t **s8;       /*!< 64 bits signed long integer pointer type points on current band*/
      uint64_t **u8;      /*!< 64 bits unsigned long integer pointer type points on current band*/
	  RGBColor **rgb;      /*!< 24 bits color (red, green, blue) pointer type */
	  BGRColor **bgr;      /*!< 24 bits color (blue, green, red) pointer type */
	};
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
	Str255 name;            /*!< Image file name */
	Str255 description;     /*!< A character string describing the image or processing performed */
	int samples;            /*!< Number of samples (pixels) per image line for each band */
	int lines;              /*!< Number of lines per image for each band */
	int bands;              /*!< Number of bands per image file */
	int headerOffset;       /*!< Refers to the number of skipped bytes of imbedded header information present in the file */
	int dataType;           /*!< Parameter identifying the type of data representation */
	int interleave;         /*!< Refers to wether the data are band sequential (BSQ), interleaved by pixel (BIP) or by line (BIL) */
	int byteOrder;          /*!< Describes the order of the bytes (only least significant byte first is supported) */
	int64_t fileSize;       /*!< Current file size of the image file in creation mode */
	int file;               /*!< Data file handle */
	int band;               /*!< Current selected band number */
	int fileType;          /*!< Input file type */
	ImageParam params;      /*!< User defined image parameters */
	void *extendedImgStructure; /*!< Useful for specific image type (like TDX, hdf5 or other) */
	double geotransform[6]; /*!< Affine transform, see getgeotransform() */ 
	int geotransformvalid;  /*!< Is affine transform valid? (false/true) */
    State_Vector svectors;  /*!< State vectors associated with image */
} *EFIDIRImage;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageByte
 * \brief BATI Image structure for 8 bits byte data type
 */
typedef struct EFIDIRImageByte_struct {
	unsigned char **p;      /*!< 8 bits byte pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageByte;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageShort
 * \brief BATI Image structure for 16 bits signed integer data type
 */
typedef struct EFIDIRImageShort_struct {
	short **p;              /*!< 16 bits signed integer pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageShort;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageInt
 * \brief BATI Image structure for 32 bits signed long integer data type
 */
typedef struct EFIDIRImageInt_struct {
	int **p;                /*!< 32 bits signed long integer pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageInt;

/**
 * \ingroup image_bati
 * \struct EFIDIRImageFloat
 * \brief BATI Image structure for 32 bits floating point data type
 */
typedef struct EFIDIRImageFloat_struct {
	float **p;              /*!< 32 bits floating point pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageFloat;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageDouble
 * \brief BATI Image structure for 64 bits double precision floating point data type
 */
typedef struct EFIDIRImageDouble_struct {
	double **p;             /*!< 64 bits double precision floating point pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageDouble;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageComplex4
 * \brief BATI Image structure for 2x16 bits integer complex data type
 */
typedef struct EFIDIRImageComplex4_struct {
	Complex4 **p;           /*!< 2x16 bits integer complex pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageComplex4;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageComplex8
 * \brief BATI Image structure for 2x32 bits complex data type
 */
typedef struct EFIDIRImageComplex8_struct {
	Complex8 **p;           /*!< 2x32 bits complex pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageComplex8;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageComplex16
 * \brief BATI Image structure for 2x64 bits double precision complex data type
 */
typedef struct EFIDIRImageComplex16_struct {
	Complex16 **p;          /*!< 2x64 bits double precision complex pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageComplex16;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageWord
 * \brief BATI Image structure for 16 bits unsigned integer data type
 */
typedef struct EFIDIRImageWord_struct {
	unsigned short **p;     /*!< 16 bits unsigned integer pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageWord;
/**
 * \ingroup image_bati
 * \struct EFIDIRImageDWord
 * \brief BATI Image structure for 32 bits unsigned long integer data type
 */
typedef struct EFIDIRImageDWord_struct {
	unsigned int **p;       /*!< 32 bits unsigned long integer pointer type */
	int nc;                 /*!< Number of image data columns */
	int nr;                 /*!< Number of image data rows */
} *EFIDIRImageDWord;

/**
 * \ingroup image_function
 * \fn EFIDIRImage alloc_image();
 * \brief Allocate an image empty structure
 *
 * \return address of the allocated image structure
 */
EFIDIRImage alloc_image();
/**
 * \ingroup image_function
 * \fn void free_image(EFIDIRImage image);
 * \brief Free a previously allocated image structure
 *
 * \param image address of the image structure
 */
void free_image(EFIDIRImage image);
/**
 * \ingroup image_function
 * \fn void alloc_image_data(EFIDIRImage image, int nc, int nr);
 * \brief Allocate image data
 *
 * \param image address of the image structure
 * \param nc the number of columns to allocate
 * \param nr the number of rows to allocate
 */
void alloc_image_data(EFIDIRImage image, int nc, int nr);
/**
 * \ingroup image_function
 * \fn void free_image_data(EFIDIRImage image);
 * \brief Free allocated image data
 *
 * \param image address of the image structure
 */
void free_image_data(EFIDIRImage image);
/**
 * \ingroup image_function
 * \fn void copy_image_data(EFIDIRImage src, EFIDIRImage dst);
 * \brief Copy the image data between two images
 *
 * \param src address of the image structure of the source image
 * \param dst address of the image structure of the destination image
 */
void copy_image_data(EFIDIRImage src, EFIDIRImage dst);
/**
 * \ingroup image_function
 * \fn void copy_image_data_at(EFIDIRImage src, EFIDIRImage dst, int offsetLin, int offsetCol);
 * \brief Copy the image data from src to dst at the given position
 *
 * \param src address of the image structure of the source image
 * \param dst address of the image structure of the destination image
 * \param offsetLin line offset of the copy in dst (from 0)
 * \param offsetCol column offset of the copy in dst (from 0)
 */
void copy_image_data_at(EFIDIRImage src, EFIDIRImage dst, int offsetLin, int offsetCol);
/**
 * \ingroup image_function
 * \fn void select_image_band(EFIDIRImage image, int band);
 * \brief Change the current image band
 *
 * Change the current image band. Re-load data is not indispensable, the access to "image->???[i][j]" directly gives acces to current band. 
 *
 * \param image address of the image structure
 * \param band the new current band number (from 0 to image->bands-1)
 */
void select_image_band(EFIDIRImage image, int band);
/**
 * \ingroup image_function
 * \fn void get_image_band(EFIDIRImage image, int band);
 * \brief return the band as a 2D array
 *
 * Return the band as a 2D array, but does not change current band
 *
 * \param image address of the image structure
 * \param band the new current band number (from 0 to image->bands-1)
 *
 * \return selected band
 */
char** get_image_band(EFIDIRImage image, int band);
/**
 * \ingroup image_function
 * \fn void set_image_param(EFIDIRImage image, const char *name, const char *value);
 * \brief set the specified image param string value
 *
 * \param image address of the image structure
 * \param name the param name
 * \param value the param string value
 */
void set_image_param(EFIDIRImage image, const char *name, const char *value);
/**
 * \ingroup image_function
 * \fn void set_image_param_long(EFIDIRImage image, const char *name, long value);
 * \brief set the specified image param long value
 *
 * \param image address of the image structure
 * \param name the param name
 * \param value the param long value
 */
void set_image_param_long(EFIDIRImage image, const char *name, long value);
/**
 * \ingroup image_function
 * \brief set the specified image param float value
 *
 * \param image address of the image structure
 * \param name the param name
 * \param value the param float value
 */
void set_image_param_float(EFIDIRImage image, const char *name, float value);
/**
 * \ingroup image_function
 * \brief set the specified image param double precision float value
 *
 * \param image address of the image structure
 * \param name the param name
 * \param value the param double precision float value
 */
void set_image_param_double(EFIDIRImage image, const char *name, double value);

/**
 * \ingroup image_function
 * \fn char *get_image_param(EFIDIRImage image, const char *name);
 * \brief get the specified image param value
 *
 * \param image address of the image structure
 * \param name the param name
 * \return the param string value or NULL if the parameter does not exist.
 */
char *get_image_param(EFIDIRImage image, const char *name);
/**
 * \ingroup image_function
 * \fn long get_image_param_long(EFIDIRImage image, const char *name);
 * \brief get the specified image param value as long
 *
 * \param image address of the image structure
 * \param name the param name
 * \return the param long value.
 */
long get_image_param_long(EFIDIRImage image, const char *name);
/**
 * \ingroup image_function
 * \brief get the specified image param value as float
 *
 * \param image address of the image structure
 * \param name the param name
 * \return the param float value.
 */
float get_image_param_float(EFIDIRImage image, const char *name);
/**
 * \ingroup image_function
 * \fn void remove_image_param(EFIDIRImage image, const char *name);
 * \brief remove the specified image param
 *
 * \param image address of the image structure
 * \param name the param name
 */
void remove_image_param(EFIDIRImage image, const char *name);
/**
 * \ingroup image_function
 * \fn void set_image_classes(EFIDIRImage image, int classes, RGBColor *classLookup, char *classNames[]);
 * \brief set the image classes
 *
 * \param image address of the image structure
 * \param classes the number of classes
 * \param classLookup the classes as color RGB triplet
 * \param classNames the classes names (can be NULL)
 */
void set_image_classes(EFIDIRImage image, int classes, RGBColor *classLookup, char *classNames[]);
/**
 * \ingroup image_function
 * \fn void get_image_classes(EFIDIRImage image, int *classes, RGBColor **classLookup, char **classNames[]);
 * \brief get the image classes
 *
 * \param image address of the image structure
 * \param classes the number of classes on return
 * \param classLookup the classes as color RGB triplet on return
 * \param classNames the classes names on return (can be NULL)
 */
void get_image_classes(EFIDIRImage image, int *classes, RGBColor **classLookup, char **classNames[]);
/**
 * \ingroup image_function
 * \fn EFIDIRImage read_image_open(const char *name);
 * \brief Open an image file for reading
 *
 * \param name the image file name
 * \return image the new image structure
 */
EFIDIRImage read_image_open(const char *name);
// TODO implement read_image_data_current_band
/*
 * \ingroup image_function
 * \fn void read_image_data_current_band(EFIDIRImage image);
 * \brief Read all the image data from the image file
 *
 * Read current band of all the image data from the image file,
 * to switch band (see select_image_band) and re-read image.
 *
 * \param image address of the image structure
 *
 * \see select_image_band, 
 */
//void read_image_data_current_band(EFIDIRImage image);
/**
 * \ingroup image_function
 * \fn void read_image_data_all_bands(EFIDIRImage image);
 * \brief Read all the image data from the image file
 *
 * Read all bands of all the image data from the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 *
 * \see select_image_band, 
 */
void read_image_data_all_bands(EFIDIRImage image);


//TODO implement read_image_line_current_band
/*
 * \ingroup image_function
 * \fn int read_image_line_current_band(EFIDIRImage image, int line);
 * \brief Read the specified line of image data from the image file
 *
 * Read current band of the specified line of image data from the image file,
 * to switch band (see select_image_band) and re-read line.
 *
 * \param image address of the image structure
 * \param line the line number
 * \return null if the specified line is outside the image range
 *
 * \see select_image_band, 
 */
//int read_image_line_current_band(EFIDIRImage image, int line);
/**
 * \ingroup image_function
 * \fn int read_image_line_all_bands(EFIDIRImage image, int line);
 * \brief Read the specified line of image data from the image file
 *
 * Read all bands of the specified line of image data from the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 * \param line the line number
 * \return null if the specified line is outside the image range
 *
 * \see select_image_band, 
 */
int read_image_line_all_bands(EFIDIRImage image, int line);

// TODO implement read_image_part_current_band
/*
 * \ingroup image_function
 * \fn int read_image_part_current_band(EFIDIRImage image, int col, int row, int nCols, int nRows);
 * \brief Read the specified part of image data from the image file
 *
 * Read current band of the specified part of image data from the image file,
 * to switch band (see select_image_band) and re-read part.
 *
 * \param image address of the image structure
 * \param col the first col of the image part
 * \param row the first row of the image part
 * \param nCols number of columns
 * \param nRows number of rows
 * \return null if the specified part is outside the image range
 *
 * \see select_image_band, 
 */
//int read_image_part_current_band(EFIDIRImage image, int col, int row, int nCols, int nRows);
/**
 * \ingroup image_function
 * \fn int read_image_part_all_bands(EFIDIRImage image, int col, int row, int nCols, int nRows);
 * \brief Read the specified part of image data from the image file
 *
 * Read all bands of the specified part of image data from the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 * \param col the first col of the image part
 * \param row the first row of the image part
 * \param nCols number of columns
 * \param nRows number of rows
 * \return null if the specified part is outside the image range
 *
 * \see select_image_band, 
 */
int read_image_part_all_bands(EFIDIRImage image, int col, int row, int nCols, int nRows);

/**
 * \ingroup image_function
 * \fn void read_image_close(EFIDIRImage image);
 * \brief Close the image file
 *
 * \param image address of the image structure
 */
void read_image_close(EFIDIRImage image);
/**
 * \ingroup image_function
 * \fn EFIDIRImage write_image_open(const char *name, const char *description, int samples, int lines, int bands, int dataType, int interleave);
 * \brief Open an image file for writing
 * 
 * \param name the image file name
 * \param description optionnal description of the image
 * \param samples number of samples
 * \param lines number of lines
 * \param bands number of bands (1 or more)
 * \param dataType type of data representation
 * \param interleave interleave method for multi-bands image
 * \return image the new image structure
 */
EFIDIRImage write_image_open(const char *name, const char *description, int samples, int lines, int bands, int dataType, int interleave);
/**
 * \ingroup image_function
 * \fn void write_image_data_all_bands(EFIDIRImage image);
 * \brief Write all the image data into the image file
 *
 * Write all bands of all the image data into the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 *
 * \see select_image_band, 
 */
void write_image_data_all_bands(EFIDIRImage image);
/**
 * \ingroup image_function
 * \fn void write_image_data_current_band(EFIDIRImage image);
 * \brief Write all the image data into the image file
 *
 * Write only the current band of all the image data into the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 *
 * \see select_image_band, 
 */
void write_image_data_current_band(EFIDIRImage image);

/**
 * \ingroup image_function
 * \fn int write_image_line_all_bands(EFIDIRImage image, int line);
 * \brief Write the specified line of image data into the image file
 *
 * Write all bands of the specified line of image data into the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 * \param line the line number
 * \return null if the specified line is outside the image range
 *
 * \see select_image_band, 
 */
int write_image_line_all_bands(EFIDIRImage image, int line);
/**
 * \ingroup image_function
 * \fn int write_image_line_current_band(EFIDIRImage image, int line);
 * \brief Write the specified line of image data into the image file
 *
 * Write only the current band of the specified line of image data into the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 * \param line the line number
 * \return null if the specified line is outside the image range
 *
 * \see select_image_band, 
 */
int write_image_line_current_band(EFIDIRImage image, int line);

/**
 * \ingroup image_function
 * \fn int write_image_part_all_bands(EFIDIRImage image, int col, int row);
 * \brief Write the specified part of image data into the image file
 *
 * Write all bands of the specified part of image data into the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 * \param col the first col of the image part
 * \param row the first row of the image part
 * \return null if the specified part is outside the image range
 *
 * \see select_image_band, 
 */
int write_image_part_all_bands(EFIDIRImage image, int col, int row);
/**
 * \ingroup image_function
 * \fn int write_image_part_current_band(EFIDIRImage image, int col, int row);
 * \brief Write the specified part of image data into the image file
 *
 * Write only the current band of the specified part of image data into the image file,
 * to switch band (see select_image_band).
 *
 * \param image address of the image structure
 * \param col the first col of the image part
 * \param row the first row of the image part
 * \return null if the specified part is outside the image range
 *
 * \see select_image_band, 
 */
int write_image_part_current_band(EFIDIRImage image, int col, int row);

/**
 * \ingroup image_function
 * \fn void write_image_close(EFIDIRImage image);
 * \brief Close the image file
 *
 * \param image address of the image structure
 */
void write_image_close(EFIDIRImage image);

/**
 * \ingroup image_function
 * \fn void export_image(EFIDIRImage image, const char *filename);
 * \brief Export and save an image
 *
 * Export an image to another format, usualy to one of the commonly used
 * like tiff, png, jpeg; determined by the extension of the filename
 * argument. Depending on the output format, more or less loss can occur.
 * use this function only for visualization purposes and very specific 
 * compatibility usecases.
 *
 * \param image address of the image structure
 * \param filename of the export
 */
void export_image(EFIDIRImage image, const char *filename);

/**
 * \ingroup image_function
 * \fn EFIDIRImage new_image(int samples, int lines, int dataType);
 * \brief Create a memory image
 * 
 * \param samples number of samples
 * \param lines number of lines
 * \param dataType type of data representation
 * \return image the new image structure
 */
EFIDIRImage new_image(int samples, int lines, int dataType);

/**
 * \ingroup image_function
 * \fn void read_classes_colormap_file(const char *name, int *classes, RGBColor **classLookup);
 * \brief read image class color table file
 * 
 * \param name is the color table file name
 * \param classes is the number of classes
 * \param classLookup is the color table
 */
void read_classes_colormap_file(const char *name, int *classes, RGBColor **classLookup);

/**
 * \ingroup image_function
 * \fn void write_classes_colormap_file(const char *name, int classes, RGBColor *classLookup);
 * \brief write image class color table file
 * 
 * \param name is the color table file name
 * \param classes is the number of classes
 * \param classLookup is the color table
 */
void write_classes_colormap_file(const char *name, int classes, RGBColor *classLookup);



#endif /*__EFIDIR_IMAGE_H__*/
