/* 
 *    Copyright (c) 2015. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup goodies
 * \defgroup date date
 * \file efidir_date.h
 * \author Matthieu Volat (ISTerre)
 * 
 * \brief Read and write dates.
 */

#ifndef __EFIDIR_DATE_H__
#define __EFIDIR_DATE_H__

/**
 * \ingroup date
 * \struct Date
 * \brief Date structure, better suited to hold the usual dates compared to
 *        C struct like timespec or tm.
 */
typedef struct date {
  int year;
  int mon;
  int day;
  int hour;
  int min;
  int sec;
  int usec;
  char *tzinfo;
} date, *Date;

/**
 * \ingroup date
 * \brief Parse a iso8601-formated date
 *
 * Parse a iso8601-formated date and create a Date structure from it. The 
 * date structure and its content is allocated in one single allocation, thus
 * it can and must be freed by a single efidir_free() call.
 *
 * \param buf  iso8601-formated date string
 *
 * \return Date structure
 */
Date parse_iso8601(const char *buf);

/**
 * \ingroup date
 * \brief Convert a Date to unix timestamp
 *
 * Convert a Date structure to a unix timestamp (number of seconds elapsed
 * since 1970 Jan 1 0:00:00), but reported as a double to preserve the
 * microseconds information.
 *
 * The date should be in UTC timezone as the current conversion mechanics
 * do not handle timezone conversions.
 *
 * \param d    Date struct to convert
 *
 * \return Unix timestamp in double-precision floating point format
 */
double date_tounix(Date d);

/**
 * \ingroup date
 * \brief Convert a Date to GPS timestamp
 *
 * Convert a Date structure to a GPS timestamp (number of seconds elapsed
 * since 1980 Jan 1 0:00:00, without leap seconds), but reported as a 
 * double to preserve the microseconds information.
 *
 * \param d    Date struct to convert
 *
 * \return GPS timestamp in double-precision floating point format
 */
double date_togps(Date d);

#endif
