/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup efidir_math_tools
 * \defgroup complex some operations of complex
 */

/**
 * \ingroup complex
 * \file efidir_complex.h
 * \author Yajing Yan and Flavien Vernier
 * 
 * \brief operator of complex.
 * 
 * file includes (stucture, prototype de fonction) of the operator complex
 * 
 * 
 */

#ifndef EFIDIR_COMPLEX_H
#define EFIDIR_COMPLEX_H

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
//#include "efidir_image.h"
//#include "efidir_param.h"
#include "efidir_math.h"

/**
 * \ingroup image_type
 * \struct Complex4
 * \brief 2x16 bits integer complex structure
 */
typedef struct Complex4 {
     short re;
     short im;
} Complex4;

/**
 * \ingroup image_type
 * \struct Complex8
 * \brief 2x32 bits complex structure
 */
typedef struct Complex8 {
     float re;
     float im;
} Complex8;

/**
 * \ingroup image_type
 * \struct Complex16
 * \brief 2x64 bits double precision complex structure
 */
typedef struct Complex16 {
     double re;
     double im;
} Complex16;

/**
 * \ingroup complex
 * \fn Complex8 cadd(Complex8 c1, Complex8 c2);
 * \brief function of calculating the sum of two Complex8 values
 *
 * \param c1 is the first Complex8 value
 * \param c2 is the second Complex8 value
 * \return a Complex8 value 
 *
 */
Complex8 cadd(Complex8 c1, Complex8 c2);

/**
 * \ingroup complex
 * \fn Complex8 csub(Complex8 c1, Complex8 c2);
 * \brief function of calculating the substraction of two Complex8 values
 *
 * \param c1 is the first Complex8 value
 * \param c2 is the second Complex8 value
 * \return a Complex8 value 
 *
 */
Complex8 csub(Complex8 c1, Complex8 c2);

/**
 * \ingroup complex
 * \fn Complex8 cmul(Complex8 c1, Complex8 c2);
 * \brief function of calculating the multiplication of two Complex8 values
 *
 * \param c1 is the first Complex8 value
 * \param c2 is the second Complex8 value
 * \return a Complex8 value 
 *
 */
Complex8 cmul(Complex8 c1, Complex8 c2);

/**
 * \ingroup complex
 * \fn Complex8 cdiv(Complex8 c1, float nb);
 * \brief function of calculating the division of a Complex8 vavue by a float number
 *
 * \param c1 is the Complex8 value
 * \param nb is the float number
 * \return a Complex8 value 
 *
 */
Complex8 cdiv(Complex8 c1, float nb);

/**
 * \ingroup complex
 * \fn Complex8 cconj(Complex8 c);
 * \brief function of calculating the conjugated Complex8 value
 *
 * \param c is the first Complex8 value
 * \return a Complex8 value 
 *
 */
Complex8 cconj(Complex8 c);

/**
 * \ingroup complex
 * \def ro(r,i) 
 * \brief return ro double value of the complex number
 *
 * ro is given by sqrt(r*r+i*i).
 *
 * \param r real value of complex number (value: int, float, double or others...) 
 * \param i imaginary value of complex number(number value: int, float, double or others...)
 *
 * \return return phi double value of the complex number
 */
#ifndef ro
#define ro(r,i)(sqrt(r*r+i*i))
#endif
// TODO error of grec letter must be replaced in sources by ro
#ifndef phi
#define phi(r,i)(sqrt(r*r+i*i))
#endif



/**
 * \ingroup complex
 * \def cos_theta(r,i) 
 * \brief return cos(theta) value of the complex number
 *
 * \param r real value of complex number (value: int, float, double or others...) 
 * \param i imaginary value of complex number(number value: int, float, double or others...)
 *
 * \return return cos(theta) double value of the complex number
 */
#ifndef cos_theta
#define cos_theta(r,i)((double)r/phi(r,i))
#endif

/**
 * \ingroup complex
 * \def sin_theta(r,i) 
 * \brief return sin(theta) value of the complex number
 *
 * \param r real value of complex number (value: int, float, double or others...) 
 * \param i imaginary value of complex number(number value: int, float, double or others...)
 *
 * \return return sin(theta) double value of the complex number
 */
#ifndef sin_theta
#define sin_theta(r,i)((double)i/phi(r,i))
#endif

/**
 * \ingroup complex
 * \def theta(r,i) 
 * \brief return theta double value of the complex number
 *
 * \param r real value of complex number (value: int, float, double or others...) 
 * \param i imaginary value of complex number(number value: int, float, double or others...)
 *
 * \return return theta double value of the complex number. theta is computed like arcos(cos_theta(r,i))
 */
#ifndef theta
#define theta(r,i)(acos(cos_theta(r,i)))
#endif

/**
 * \ingroup complex
 * \def real(p,t) 
 * \brief return real double value of the complex number
 *
 * \param p phi value of complex number (value: int, float, double or others...) 
 * \param t theta value of complex number(number value: int, float, double or others...)
 *
 * \return return real double value of the complex number. 
 */
#ifndef real
#define real(p,t) (p*cos(t))
#endif

/**
 * \ingroup complex
 * \def imag(p,t) 
 * \brief return imaginary double value of the complex number
 *
 * \param p phi value of complex number (value: int, float, double or others...) 
 * \param t theta value of complex number(number value: int, float, double or others...)
 *
 * \return return imaginary double value of the complex number. 
 */
#ifndef imag
#define imag(p,t) (p*sin(t))
#endif

/**
 * \ingroup complex
 * \def arg(r,i)
 * \brief return argument of complex number
 *
 * \param r real part of complex number (value: int, float, double or others...) 
 * \param i imaginary part of complex number (value: int, float, double or others...)
 *
 * \return return 
 */
#ifndef arg
#define arg(r,i) (atan2(i,r))
#endif

#endif
