/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * \ingroup efidir 
 * \defgroup atmosphere_manager Atmosphere manager
 * \file efidir_atmosphere.h
 * \author Flavien Vernier (LISTIC)
 * 
 * \brief EFIDIR atmosphere manager
 */

#ifndef __EFIDIR_ATMOSPHERE_H__
#define __EFIDIR_ATMOSPHERE_H__

#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <string.h>

#include "efidir_file.h"
#include "efidir_file_txt.h"
#include "efidir_assert.h"
#include "efidir_allocation.h"
#include "efidir_boolean.h"
#include "efidir_matrix.h"

enum dimension {
  NO_DIM,
  D2,
  D3,
};

enum file_type {
  NO_FILE_TYPE,
  TXT_NARR,
  BIN_EFIDIR,
};
/**
 * \ingroup atmosphere_type
 * \struct AtmosphereParam
 * \brief Atmosphere parameter structure
 */
typedef struct AtmosphereParam {
	char *name;              /*!> Parameter name */
	char *value;             /*!> Parameter string value */
	struct AtmosphereParam *next; /*!> Next parameter in the parameters list */
} *AtmosphereParam;

/* typedef struct AtmosphereData{ */
/*   union { */
/*     float** values2D; */
/*     float*** values3D; */
/*   }; */
/*   char *type; */
/*   struct AtmosphereData *next; */
/* }atmosphereData, *AtmosphereData; */

/**
 * \ingroup atmosphere_manager
 * \struct AtmosphereDataSeries
 * \brief EFIDIR Atmosphere data structure
 */
typedef struct AtmosphereDataSeries{
  union {
    double** values2D;  /*!< 2D matrix for 2m data */
    double*** values3D; /*!< 3D matrix for data */
  };
  //  AtmosphereData data;
  struct AtmosphereDataSeries *next; /*!< Data of next date */ 
}atmosphereDataSeries, *AtmosphereDataSeries;


/**
 * \ingroup atmosphere_manager
 * \struct EFIDIRAtmosphere
 * \brief EFIDIR Atmosphere structure
 */
typedef struct EFIDIRAtmosphere {
  AtmosphereDataSeries series; /*!< Temporal vector of data */
  char *type;             /*!< Name of data type */
  int nc;                 /*!< Number of loaded atmosphere data columns (longitudes)*/
  int nr;                 /*!< Number of loaded atmosphere data rows (latitudes)*/
  int nl;                 /*!< Number of loaded atmosphere data levels */
  int nt;                 /*!< Number of loaded atmosphere data times */
  int nD;                 /*!< 2D or 3D*/
  int columns;            /*!< Number of columns (longitudes) in atmosphere file */
  int lines;              /*!< Number of lines (latitudes) in atmosphere file */
  int nbr_levels;             /*!< Number of levels in atmosphere file */
  int nbr_times;              /*!< Number of date in atmosphere file */
  int nc0;
  int nr0;
  int nl0;
  int nt0;
  float *longitudes;      /*!< Longitude values*/
  float lon0;             /*!< Initial longitude value */
  float lonStep;          /*!< Step in longitude between two points (0 if step is not constant)*/
  float *latitudes;       /*!< Latitude values */
  float lat0;             /*!< Initial latitude value */
  float latStep;          /*!< Step in latitude between two points (0 if step is not constant)*/
  float *times;           /*!< Time values */
  float time0;            /*!< Initial date value */
  float timeStep;         /*!< Step in date between two points (0 if step is not constant)*/
  float *levels;          /*!< Date values */
  float level0;           /*!< Initial level value */
  float levelStep;        /*!< Step in level between two points (0 if step is not constant)*/
  char* name;             /*!< Name of data file */
  int file;
  int file_type;
  int64_t fileSize;
  int byteOrder;
  char* description;
  AtmosphereParam params;
} *EFIDIRAtmosphere;



/*
 * \ingroup atmosphere_manager
 * \fn EFIDIRAtmosphere download_atmosphere();
 * \brief Open an atmosphere file for reading
 *
 * \param name the atmosphere file name
 * \return atmosphere the new atmosphere structure
 */
EFIDIRAtmosphere download_atmosphere();

/**
 * \ingroup atmosphere_manager
 * \fn EFIDIRAtmosphere read_atmosphere_open(const char *name);
 * \brief Open an atmosphere file for reading
 *
 * \param name the atmosphere file name
 * \return atmosphere the new atmosphere structure
 */
EFIDIRAtmosphere read_atmosphere_open(const char *name);

/**
 * \ingroup atmosphere_manager
 * \fn void read_atmosphere_data_point(EFIDIRAtmosphere atmosphere, int line, int column);
 * \brief load a data of each level for a given point.
 *
 *
 *
 * \param atmosphere address of the atmosphere structure
 * \param line
 * \param column
 */
void read_atmosphere_data_point(EFIDIRAtmosphere atmosphere, int line, int column);

/**
 * \ingroup atmosphere_manager
 * \fn void read_atmosphere_data_line(EFIDIRAtmosphere atmosphere, int line);
 * \brief load a section defined by a line
 *
 *
 *
 * \param atmosphere address of the atmosphere structure
 * \param line line number that define the section
 */
void read_atmosphere_data_line(EFIDIRAtmosphere atmosphere, int line);

/**
 * \ingroup atmosphere_manager
 * \fn void read_atmosphere_data_column(EFIDIRAtmosphere atmosphere, int column);
 * \brief load a section defined by a column
 *
 *
 *
 * \param atmosphere address of the atmosphere structure
 * \param column culumn number that define the section
 */
void read_atmosphere_data_column(EFIDIRAtmosphere atmosphere, int column);

/**
 * \ingroup atmosphere_manager
 * \fn void read_atmosphere_data_level(EFIDIRAtmosphere atmosphere, int level);
 * \brief load a section defined by a level
 *
 *
 *
 * \param atmosphere address of the atmosphere structure
 * \param level level number that define the section
 */
void read_atmosphere_data_level(EFIDIRAtmosphere atmosphere, int level);

/**
 * \ingroup atmosphere_manager
 * \fn void read_atmosphere_data_part(EFIDIRAtmosphere atmosphere, int level, int line, int column, int nbLevels, int nbLines, int nbColumns);
 * \brief  Load all dates of a part of data file.
 *
 *
 * \param atmosphere address of the atmosphere structure
 * \param level 
 * \param column
 * \param line 
 * \param nbLevels
 * \param nbLines
 * \param nbColumns
 */
void read_atmosphere_data_part(EFIDIRAtmosphere atmosphere, int level, int line, int column, int nbLevels, int nbLines, int nbColumns);

/**
 * \ingroup atmosphere_manager
 * \fn void read_atmosphere_data(EFIDIRAtmosphere atmosphere);
 * \brief  Load all dates, all levels, all lines and all columns of data file.
 *
 *
 * \param atmosphere address of the atmosphere structure
 */
void read_atmosphere_data(EFIDIRAtmosphere atmosphere);
/*
 * WRITE
 */
/**
 * \ingroup atmosphere_manager
 * \fn EFIDIRAtmosphere write_atmosphere_open(const char *name, const char *description, int nbr_levels, int nbr_lines, int nbr_columns)
 * \brief  Open an atmosphre to write into
 *
 * Open an atmosphre to write into. This atmosphere can store only one time step.
 *
 * \param name atmosphere file name
 * \param description comment about atmosphere
 * \param nbr_levels number of levels in the atmosphere
 * \param nbr_lines  number of lines in the atmosphere
 * \param nbr_columns number of columns in the atmosphere
 *
 * \return an empty atmosphere with nbr_levels levels, nbr_lines lines and nbr_columns columns
 */
EFIDIRAtmosphere write_atmosphere_open(const char *name, const char *description, int nbr_levels, int nbr_lines, int nbr_columns);

/**
 * \ingroup atmosphere_manager
 * \fn void alloc_atmosphere_data(EFIDIRAtmosphere atmosphere, int nl, int nr, int nc);
 */
void alloc_atmosphere_data(EFIDIRAtmosphere atmosphere, int nl, int nr, int nc);

/**
 * \ingroup atmosphere_manager
 * \fn void write_atmosphere_data(EFIDIRAtmosphere atmosphere);
 * \brief  Save the atmosphere data in file
 *
 *
 * \param atmosphere the atmosphere to save
 */
void write_atmosphere_data(EFIDIRAtmosphere atmosphere);

/**
 * \ingroup atmosphere_manager
 * \fn void write_atmosphere_data_line(EFIDIRAtmosphere atmosphere, int line);
 * \brief  Save the atmosphere data in file at the given line
 *
 *
 * \param atmosphere the atmosphere to save
 * \param line define the line where the data are saved
 */
void write_atmosphere_data_line(EFIDIRAtmosphere atmosphere, int line);

/**
 * \ingroup atmosphere_manager
 * \fn void write_atmosphere_data_column(EFIDIRAtmosphere atmosphere, int column);
 * \brief  Save the atmosphere data in file at the given column
 *
 *
 * \param atmosphere the atmosphere to save
 * \param column define the column where the data are saved
 */
void write_atmosphere_data_column(EFIDIRAtmosphere atmosphere, int column);

/**
 * \ingroup atmosphere_manager
 * \fn void write_atmosphere_data_level(EFIDIRAtmosphere atmosphere, int level);
 * \brief  Save the atmosphere data in file at the given level
 *
 *
 * \param atmosphere the atmosphere to save
 * \param level define the level where the data are saved
 */
void write_atmosphere_data_level(EFIDIRAtmosphere atmosphere, int level);

/**
 * \ingroup atmosphere_manager
 * \fn void write_atmosphere_data_part(EFIDIRAtmosphere atmosphere, int level, int line, int column, int nbLevels , int nbLines, int nbColumns);
 * \brief  Save the atmosphere data in file at the given area
 *
 *
 * \param atmosphere the atmosphere to save
 * \param level define the first level of the area
 * \param line define the first line of the area
 * \param column define the first column of the area
 * \param nbLevels define the number of levels of the area
 * \param nbLines define the number of lines of the area
 * \param nbColumns define the number of columns of the area
 */
void write_atmosphere_data_part(EFIDIRAtmosphere atmosphere, int level, int line, int column, int nbLevels , int nbLines, int nbColumns);

/**
 * \ingroup atmosphere_manager
 * \fn void write_atmosphere_close(EFIDIRAtmosphere atmosphere);
 * \brief  Close an EFIDIRAtmosphere opened for writing
 *
 * \param atmosphere the atmosphere to close
 *
 * Write the header file of EFIDIRAtmosphere opened for writing and cLeanup all alocated memory for this atmosphere.
 */
void write_atmosphere_close(EFIDIRAtmosphere atmosphere);

/*
* GOODIES
*/

/**
 * \ingroup atmosphere_manager
 * \fn void read_atmosphere_close(EFIDIRAtmosphere atmosphere);
 * \brief Close an atmosphere file open for reading
 *
 * \param atmosphere address of the atmosphere structure
 */
void read_atmosphere_close(EFIDIRAtmosphere atmosphere);


/**
 * \ingroup atmosphere_manager
 * \fn int longitude2index(EFIDIRAtmosphere atmosphere, float longitude);
 * \brief Give the index corresponding to a longitude.
 *
 * Give the index, in the data file, of a given longitude value.
 * Be careful this index can be different of the index of loaded values.
 *
 * \param atmosphere address of the atmosphere structure
 * \param longitude longitude of the requested index
 *
 * \return index of the requested longitude
 */
int longitude2index(EFIDIRAtmosphere atmosphere, float longitude);
/**
 * \ingroup atmosphere_manager
 * \fn int latitude2index(EFIDIRAtmosphere atmosphere, float latitude);
 * \brief Give the index corresponding to a latitude.
 *
 * Give the index, in the data file, of a given latitude value.
 * Be careful this index can be different of the index of loaded values.
 *
 * \param atmosphere address of the atmosphere structure
 * \param latitude latitude of the requested index
 *
 * \return index of the requested latitude
 */
int latitude2index(EFIDIRAtmosphere atmosphere, float latitude);
/**
 * \ingroup atmosphere_manager
 * \fn int pression2index(EFIDIRAtmosphere atmosphere, float pression);
 * \brief Give the index corresponding to a pressure.
 *
 * Give the index, in the data file, of a given pressure value.
 * Be careful this index can be different of the index of loaded values.
 *
 * \param atmosphere address of the atmosphere structure
 * \param pression pressure of the requested index
 *
 * \return index of the requested pressure
 */
int pression2index(EFIDIRAtmosphere atmosphere, float pression);
/**
 * \ingroup atmosphere_manager
 * \fn int time2index(EFIDIRAtmosphere atmosphere, float time);
 * \brief Give the index corresponding to a time.
 *
 * Give the index, in the data file, of a given time value.
 * Be careful this index can be different of the index of loaded values.
 *
 * \param atmosphere address of the atmosphere structure
 * \param time time of the requested index
 *
 * \return index of the requested time
 */
int time2index(EFIDIRAtmosphere atmosphere, float time);



/**
 * \ingroup atmosphere_manager
 * \fn void copy_meta_data(EFIDIRAtmosphere atmosphereIn, EFIDIRAtmosphere atmosphereOut);
 * \brief Copy meta data from an atmosphere structure to another.
 *
 * Copy meta data from atmosphereIn to atmosphereOut.
 *
 * \param atmosphereIn address of the atmosphere structure
 * \param atmosphereOut address of the atmosphere structure
 *
 */
void copy_meta_data(EFIDIRAtmosphere atmosphereIn, EFIDIRAtmosphere atmosphereOut);

double **get2DvaluesAt(EFIDIRAtmosphere atmosphere, float time);
double ***get3DvaluesAt(EFIDIRAtmosphere atmosphere, float time);



#endif //__EFIDIR_ATMOSPHERE_H__
