/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 
#ifndef COM_ZEROPAD_H
#define COM_ZEROPAD_H

#include "efidir_image.h"
#include "efidir_image_info.h"
#include "efidir_param.h"
#include "efidir_string.h"
#include <stdlib.h> 

/**
 * \ingroup com_resize_image
 * \defgroup zeropad Image zero padding.
 * \file com_zeropad.h
 * \author Matthieu Volat (ISTerre)
 *
 * \brief Increase an image size with zeroes.
 */

/**
 * \ingroup zeropad
 * \struct zeropad_param
 *
 * \brief structure that contains the parameters of the operator zeropad
 */
typedef struct struct_zeropad_param{
	int ncols;	/*!< Output number of columns */
	int nrows;	/*!< Output number of rows */
	int xoff;	/*!< Input image x offset */
	int yoff;	/*!< Input image y offset */
} zeropad_param, *Zeropad_param;

/**
 * \ingroup zeropad
 * \fn new_zeropad_param()
 *
 * \return A new reference on allocated zeropad_param structure
 *
 * \brief Create a new Zeropad_param reference 
 */
Zeropad_param new_zeropad_param();

/**
 * \ingroup zeropad
 * \fn free_zeropad_param(Zeropad_param p)
 *
 * \param p reference on a zeropad_param structure
 *
 * \brief Free an allocated Zeropad_param reference 
 */
void free_zeropad_param(Zeropad_param p);

/**
 * \ingroup zeropad
 * \fn void define_zeropad_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 *        the same parameter (NULL is accepted)
*/
void define_zeropad_param(char *extra_desc);

/**
 * \ingroup zeropad
 * \fn Zeropad_param get_zeropad_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Zeropad_param that contains the parameters.
 */
Zeropad_param get_zeropad_param();

/**
 * \ingroup zeropad
 * \brief Zeropad an image
 *
 * The zeropad operator resize an image, often enlarge it, filling the extended
 * area with zeroes. The operator allows to offset the original image position
 * in the output image, but only with positive offsets for now.
 *
 * Parts of the image that do not fit into the new geometry are cropped.
 *
 * Zeropad will fill the following fields, updated, in the image header:
 *  - x start, y start: offseted with xoff, yoff
 *  - image info: updated in respect of the geographic representation system
 *
 * \param p    Structure of the parameters for the operator
 * \param in   Image to be zeropadped, Image data can be not loaded
 * \param out  Image of result
 */
void zeropad(Zeropad_param p, EFIDIRImage in, EFIDIRImage out);

#endif /* ZEROPAD_H */
