/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

#ifndef COM_WRAP_H
#define COM_WRAP_H

#include "efidir_image.h"
#include "efidir_param.h"
#include <math.h>
#include <stdlib.h> 

/**
 * \ingroup common_operators
 * \defgroup wrap Wrap the image value
 */
/**
 * \ingroup wrap
 * \typedef Wrap_param
 * \author Yajing Yan (LISTIC)
 *
 * \brief reference on wrap_param structure
 *
 */

/**
 * \ingroup wrap
 * \enum dataindex constant definition of input image data type
 * \brief data type of image
 */
enum dataindex{
  phase = 1, /*!< interferometric phase in radian*/
  displacement = 2  /*!< displacement in meter*/
};

/**
 * \ingroup wrap
 * \struct wrap_param
 *
 * \brief structure that contains the parameters of the operator wrap
 *
 */
typedef struct struct_wrap_param{
  int type; /*!< image data type */
  float wavelength; /*!< wavelength of sensor */
  float amp; /*!< amplitude of noise */
}wrap_param, *Wrap_param;
/**
 * \ingroup wrap
 * \fn new_wrap_param()
 *
 * \return A new reference on allocated wrap_param structure
 *
 * \brief Create a new Wrap_param reference 
 *
 */
static Wrap_param new_wrap_param();
/**
 * \ingroup wrap
 * \fn free_wrap_param(Wrap_param p)
 *
 * \param p A reference on a wrap_param structure
 *
 * \brief Free an allocated Wrap_param reference 
 *
 */
void free_wrap_param(Wrap_param p);

/**
 * \ingroup wrap
 * \fn void define_wrap_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/
void define_wrap_param(char *extra_description);
/**
 * \ingroup wrap
 * \fn Wrap_param get_wrap_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Wrap_param that contains the parameters.
 */
Wrap_param get_wrap_param();

/**
 * \ingroup wrap
 * \fn void wrap(Wrap_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief Wrap the phase
 * Wrap the interferometric phase
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImage Image to be wrapped.
 * \param outputImage is the image of result
 *
 */
void wrap(Wrap_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);

/**
 * \ingroup wrap
 * \fn float gaussian(float sigma, int iuse);
 * \brief gaussian noise generator
 * Function allowing to generate random noise with a gaussian distribution
 *
 * \param sigma is the standard deviation of noise
 * \param iuse is an index
 * \return a value of noise whose distribution is gaussian
 *
 */
float gaussian(float sigma, int iuse);


/**
 * \ingroup wrap
 * \fn void adnoise(Wrap_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief add gaussian noise to image data
 * Function allowing to add gaussian noise to image data
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImage Image to be noised
 * \param outputImage is the image of result
 *
 */
void adnoise(Wrap_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);

/**
 * \ingroup wrap
 * \fn void wrap_with_noise(Wrap_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief Wrap the phase with gaussien noise additionned
 * Wrap the interferometric phase in additionning gaussien noise
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImage Image to be wrapped.
 * \param outputImage is the image of result
 *
 */
void wrap_with_noise(Wrap_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);

#endif
