/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup display
 * \defgroup width_ori Calculate the width and the orientation of the interferometric fringes
 */
/**
 * \ingroup width_ori
 * \file com_width_ori.h
 * \author Yajing Yan
 * 
 * \brief operator of estimation of width and orientation of fringes in the interferograms
 * 
 * file includes (stucture, prototype de fonction) of the operator width_ori
 * 
 * 
 */

#ifndef COM_WIDTH_ORI_H
#define COM_WIDTH_ORI_H

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_math.h"

/**
 * \ingroup width_ori
 * \typedef Width_ori_param
 *
 * \brief reference on width_ori_param structure
 *
 */
/**
 * \ingroup width_ori
 * \struct width_ori_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_width_ori_param{
  int x_first; /*!< first column to calculate */
  int y_first;  /*!< first line to calculate */
  int nb_x; /*!< number of columns to calculate */
  int nb_y; /*!< number of lines to calculate */
}width_ori_param, *Width_ori_param;

/**
 * \ingroup width_ori
 * \fn new_width_ori_param()
 *
 * \return A new reference on allocated width_ori_param structure
 *
 * \brief Create a new Width_ori_param reference 
 *
 */
static Width_ori_param new_width_ori_param();
/**
 * \ingroup width_ori
 * \fn free_width_ori_param(Width_ori_param p)
 *
 * \param p A reference on a width_ori_param structure
 *
 * \brief Free an allocated Width_ori_param reference 
 *
 */
void free_width_ori_param(Width_ori_param p);


/**
 * \ingroup width_ori
 * \fn void define_width_ori_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_width_ori_param(char *extra_description);

/**
 * \ingroup width_ori
 * \fn Width_ori_param get_width_ori_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Width_ori_param that contains the parameters.
 */
Width_ori_param get_width_ori_param();

/**
 * \ingroup width_ori
 * \fn void width_ori(Width_ori_param p, EFIDIRImage inputImage, EFIDIRImage outputWidth, EFIDIRImage outputOrientation);
 * \brief function of calculating the width and orientation of fringes
 *
 * \param p is the structure parameter for the operator
 * \param inputImage is the input complex image of frequency
 * \param outputWidth is the image of width of fringes
 * \param outputOrientation is the image of orientation of fringes
 *
 */
void width_ori(Width_ori_param p, EFIDIRImage inputImage, EFIDIRImage outputWidth, EFIDIRImage outputOrientation);

/**
 * \ingroup width_ori
 * \fn void width_ori_mask(Width_ori_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage outputWidth, EFIDIRImage outputOrientation);
 * \brief function of calculating the width and orientation of fringes
 *
 * \param p is the structure parameter for the operator
 * \param inputImage is the input complex image of frequency
 * \param inputMask is the mask of input image
 * \param outputWidth is the image of width of fringes
 * \param outputOrientation is the image of orientation of fringes
 *
 */
void width_ori_mask(Width_ori_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage outputWidth, EFIDIRImage outputOrientation);

#endif
