
/*
 *  Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *  This file is part of EFIDIR tools.
 *
 *  EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  EFIDIR tool(s) is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public licence
 *  along with EFIDIR tools. If not, see <http://www.gnu.org/licenses/>.
 */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "efidir_allocation.h"

/**
 * \ingroup mosaic
 * \defgroup warp_matrix Manipulation of matrix to get a new image from several images with common areas
 */

#ifndef __WARP_MATRIX_H__
#define __WARP_MATRIX_H__


/**
 * \ingroup mosaic
 * \file com_warp_matrix.h
 * \author Vincent Laugier and Jean-Denis Coffre
 * \brief Several functions to get a new image from several images with common areas
 */

/**
 * \ingroup mosaic
 * \struct warp_matrix
 * \brief Structure containing the parameters for warp matrix
 */

typedef struct struct_matrix {
int n, m;
double **c;
} warp_matrix, *Warp_Matrix; 

/**
 * \ingroup mosaic
 * \fn Warp_Matrix new_warp_matrix(int n ,int m);
 * \brief Allocate a Warp_Matrix structure
 *
 * \param n is the matrix parameter
 * \param m is the matirx parameter
 *
 * \return an allocated Warp_Matrix structure
 */
Warp_Matrix new_warp_matrix(int n ,int m);

/**
 * \ingroup mosaic
 * \fn void free_warp_matrix(Warp_Matrix mat);
 * \brief Free a previously allocated Warp_Matrix structure
 *
 * \param mat is the matrix structure to be freed.
 *
 */

void free_warp_matrix(Warp_Matrix mat);

/**
 * \ingroup mosaic
 * \fn Warp_Matrix warp_matrix_product(Warp_Matrix m1, Warp_Matrix m2);
 * \brief Calculation of the product of 2 matrices
 *
 * \param m1 is the first matrix
 * \param m2 is the second matrix
 *
 * \return the product matrix
 */
Warp_Matrix warp_matrix_product(Warp_Matrix m1, Warp_Matrix m2);


/**
 * \ingroup mosaic
 * \fn Warp_Matrix warp_matrix_transpose(Warp_Matrix m1);
 * \brief Calculation of the transposed matrix
 *
 * \param m1 is the matrix whose transposed matrix we need
 *
 * \return the transposed matrix
 */
Warp_Matrix warp_matrix_transpose(Warp_Matrix m1);


/**
 * \ingroup mosaic
 * \fn double warp_matrix_det(Warp_Matrix mat);
 * \brief Calculation of the determinant of a matrix
 *
 * \param mat is the matrix to be calculated
 *
 * \return the determinant value 
 */
double warp_matrix_det(Warp_Matrix mat);


/**
 * \ingroup mosaic
 * \fn Warp_Matrix warp_matrix_com(Warp_Matrix m1);
 * \brief Adjoint function for calculation of inverse matrix
 *
 * \param m1 is the matrix to be calculated
 *
 * \return A Warp_Matrix structure
 */
Warp_Matrix warp_matrix_com(Warp_Matrix m1);


/**
 * \ingroup mosaic
 * \fn Warp_Matrix warp_matrix_inv(Warp_Matrix m1);
 * \brief Calculation of inverse matrix
 *
 * \param m1 is the matrix to be calculated
 *
 * \return the inversed matrix
 */
Warp_Matrix warp_matrix_inv(Warp_Matrix m1);

/**
 * \ingroup mosaic
 * \fn Warp_Matrix warp_matrix_restrict(Warp_Matrix m1, int r, int c);
 * \brief Adjoint function for calculation of inverse matrix
 *
 * \param m1 is the matrix to be calculated
 * \param r is the number of lines
 * \param c is the number of columns
 *
 * \return the determinant value 
 */

Warp_Matrix warp_matrix_restrict(Warp_Matrix m1, int r, int c);

#endif
