/*
 *  Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *  This file is part of EFIDIR tools.
 *
 *  EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  EFIDIR tool(s) is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public licence
 *  along with EFIDIR tools. If not, see <http://www.gnu.org/licenses/>.
 */

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "efidir_image.h"
#include "efidir_allocation.h"
#include "com_warp_matrix.h"

/**
 * \ingroup mosaic
 * \defgroup wrap_image Get a new image from several images with common areas
 */

/**
 * \ingroup mosaic
 * \file com_warp_image.h
 * \author Vincent Laugier and Jean-Denis Coffre
 * \brief Several functions to get a new image from several images with common areas
*/

#ifndef __WARP_IMAGE_H__
#define __WARP_IMAGE_H__

/**
 * \ingroup mosaic
 * \struct Coef_matrix
 * \brief Structure containing the coeficients of both polynoms which reflect the transformation from one image to another.

	The structure contains the polynomial degree (n) and polynomial coefficients P and Q where :
	x' = a(x,y) = S(S( P[i][j] * x^i * y^j ))

	where S is the sum from 0 to n

	Where x' and y' are the locations in the warp image, x and y are the locations in the base image, n is the polynomial degree, and P and Q are the polynomial coefficients. The P and Q polynomial coefficients matrices are written to the file by rows, one element per line. For example, a 2 x 2 matrix of P and Q is written in the following format:
	
	P[0, 0]  P[1, 0] 
	P[0, 1]  P[1, 1] 
	
	Q[0, 0]  Q[1, 0] 
	Q[0,1]  Q[1, 1] 

*/
typedef struct struct_coef_matrix {
 int n;
 double **p_coef;
 double **q_coef;
} coef_matrix, *Coef_matrix;



/**
 * \ingroup mosaic
 * \fn Coef_matrix new_coef_matrix(int n);
 * \brief Allocate a Coef_matrix structure
 *
 * \return address of the allocated Coef_matrix structure
 */
Coef_matrix new_coef_matrix(int n);

/**
 * \ingroup mosaic
 * \fn void free_coef_matrix(Coef_matrix matrix);
 * \brief Free a previously allocated Coef_matrix structure
 *
 * \param matrix address of the Coef_matrix structure
 */
void free_coef_matrix(Coef_matrix matrix);

/**
 * \ingroup mosaic
 * \fn Coef_matrix new_identity_coef_matrix(int degree);
 * \brief Create a new Coef_matrix reflecting no transformation (x' = x ; y' = y)
 *
 * \param degree the degree of the two matrices
 */
Coef_matrix new_identity_coef_matrix(int degree);


/**
 * \ingroup mosaic
* \struct struct_image_warp
* \brief image structure with relocating information
*/
typedef struct struct_image_warp {
 union{
 	EFIDIRImage image;
 };
	Coef_matrix cm_direct;
	Coef_matrix cm_indirect; 


} image_warp, *Image_warp;
/**
 * \ingroup mosaic
*\fn Image_warp new_image_warp(EFIDIRImage an_image, Coef_matrix cm_direct, Coef_matrix cm_indirect)
*\brief create an Image_warp structure
*\param an_image EFIDIRImage - the warp image
*\param cm_direct Coef_matrix - transformation from this warp to the reference image
*\param cm_indirect Coef_matrix - transformation from the reference image to this warp
*\return an Image_warp with location information thank to cm
*/
Image_warp new_image_warp(EFIDIRImage an_image, Coef_matrix cm_direct, Coef_matrix cm_indirect);

/**
 * \ingroup mosaic
*\fn void free_image_warp(Image_warp image_w)
*\brief delete an Image_warp structure
*\param image_w Image_warp - image to delete
*\return void
*/
void free_image_warp(Image_warp image_w);

/**
 * \ingroup mosaic
*\fn double * eval_polynome(double x, double y, Coef_matrix cm)
*\brief evaluate a polynome based on Coef_matrix conventions
*\param x double - first variable of the polynome
*\param y double - second variable of the polynome
*\param cm .Coef_matrix - matrix with polynome's coefficients
*\return the two value of the polynome
*/
double * eval_polynome(double x, double y, Coef_matrix cm);

/**
 * \ingroup mosaic
 * \fn void relocate_images(Image_warp image_ref ,Image_warp* image_table, int nb)
 * \brief relocate images regarding the global image
 * \param image_ref Image_warp - 
 * \param image_table Image_warp* - a table of image_warp
 * \param nb int - the number of images in the table
 */
void relocate_images(Image_warp image_ref ,Image_warp* image_table, int nb);


/**
 * \enum Point_Agregation_Method
 * \brief Point agregation Method 
*/
enum Point_Agregation_Method {
	PAM_AVERAGE,
	PAM_AVERAGE_EXCEPT_REF,
	PAM_FIRST_NON_NULL
};




/**
*\fn int image_mix_to_file(Image_warp image_ref, Image_warp * image_w_table, int nb_image_w ,char * file_path, int auto_first_corn, int x_start, int y_start, int nb_samples, int nb_lines, int bilinear_interpolation_bool, int Point_Agregation_Method);
*\brief Create a new file resulting of the mix of many images. A partial traitment can be done (write only a partial area of the total mixed image), also a bilinear_interpolation or not. At this point, only floats (DATA_TYPE_FLOAT) are managed.The process is done line by line and the size of the final image doesn't matter : this function has been done to prevent any memory allocation problem.
*\param  image_ref Image_warp - The base image
*\param  image_w_table Image_warp * - warp images array
*\param  nb_image_w int - Size of image_w_table
*\param  file_path Char * - the path of the target file
*\param  auto_first_corn int - Define whether the top-left corner will be calculed automatically or if the picture starts with user's indications
*\param  x_start int - column index of the new image relatively to the first column of the reference image  
*\param  y_start int - line index of the new image relatively to the first line of the reference image  
*\param  nb_samples int - number of columns to process (use 0 to process until the end) 
*\param  nb_lines int - number of lines to process (use 0 to process until the end)
*\param  bilinear_interpolation_bool int - Define whether the function will determine each new point with a bilinear interpolation of its coordinates (true) or not (false)
*\param  Point_Agregation_Method int - Define the way the points will be acquired (0 : average ; 1 : average except reference ; 2 : first point found)
*\return 1 if the function is successful, otherwise 0. 
*/
int image_mix_to_file(Image_warp image_ref, Image_warp * image_w_table, int nb_image_w ,char * file_path, int auto_first_corn, int x_start, int y_start, int nb_samples, int nb_lines, int bilinear_interpolation_bool, int Point_Agregation_Method);

/**
*\fn int image_mix(Image_warp image_ref, Image_warp * image_w_table, int nb_image_w ,EFIDIRImage output_image, int auto_first_corn, int x_start, int y_start, int nb_samples, int nb_lines, int bilinear_interpolation_bool, int Point_Agregation_Method);
*\brief Create a new EFIDIRImage resulting of the mix of many images. A partial traitment can be done (load only a partial area of the total mixed image), also a bilinear_interpolation or not. At this point, only floats (DATA_TYPE_FLOAT) are managed. The bigger the final image will be, the more memory the function will need to allocate. As a consequence, if the final image is predicted to be big, image_mix_to_file will be more appropriated.
*\param  image_ref Image_warp - The base image
*\param  image_w_table Image_warp * - warp images array
*\param  output_image EFIDIRImage - the target EFIDIRImage
*\param  nb_image_w int - Size of image_w_table
*\param  auto_first_corn int - Define whether the top-left corner will be calculed automatically or if the picture starts with user's indications
*\param  x_start int - column index of the new image relatively to the first column of the reference image  
*\param  y_start int - line index of the new image relatively to the first line of the reference image  
*\param  nb_samples int - number of columns to process (use 0 to process until the end) 
*\param  nb_lines int - number of lines to process (use 0 to process until the end)
*\param  bilinear_interpolation_bool int - Define whether the function will determine each new point with a bilinear interpolation of its coordinates (true) or not (false)
*\param  Point_Agregation_Method int - Define the way the points will be acquired (0 : average ; 1 : average except reference ; 2 : first point found)
*\return 1 if the function is successful, otherwise 0. 
*/
int image_mix(Image_warp image_ref, Image_warp * image_w_table, int nb_image_w ,EFIDIRImage output_image, int auto_first_corn, int x_start, int y_start, int nb_samples, int nb_lines, int bilinear_interpolation_bool, int Point_Agregation_Method);


float * get_points(int x, int y, Image_warp image_ref, Image_warp * image_w_table,int nb_image_w , int bilinear_interpolation_bool);
float get_point(int x, int y, Image_warp image_ref, Image_warp * image_w_table,int nb_image_w , int bilinear_interpolation_bool, int Point_Agregation_Method);

float bilinear_approx(double * coordinated, Image_warp image_w);

void save_image_warp_transformation(char * base_name, Image_warp imw, char * warp_name);

#endif
