/*  Copyright (c) 2008. The EFIDIR team. All right reserved.
* 
*  This file is part of EFIDIR tools.
* 
*  EFIDIR tool(s) is free software: you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
* 
*  EFIDIR tool(s) is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
* t
*  You should have received a copy of the GNU General Public License
* along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
* 
*  Any information about the EFIDIR project can be found on www.efidir.fr
*/


#ifndef COM_TEMPORAL_STAT_H
#define COM_TEMPORAL_STAT_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_math.h"
#include "efidir_image_list.h"
#include "efidir_param.h"


/**
 * \ingroup statistic
 * \defgroup temporal_stat Average of amplitude, intensity and logarithm of image series
 */
/**
 * \ingroup temporal_stat
 * \author Sophie Martz (CNRS)
 *
 */
/**
 * \ingroup temporal_stat
 * \struct temporal_stat_param
 *
 * \brief structure that contains the parameters of the operator temporal_stat 
 *
 */
typedef struct Temporal_stat_param
{
  int stat_op; /*!< 0 = the intensity average and standard deviation || 1 = (Intensity average)sqrt and Standard dev of int_average || 2 = the amplitude average and standard deviation || 3 = the Logarithm average and standard deviation || 4 = exponential of Logartihm average and standard deviation of log_average*/
  float threshold; /*!< optional parameter for the minimum value for the calculation of each average */
  float *coeff_calib;
}temporal_stat_param, *Temporal_stat_param;

/**
 * \ingroup temporal_stat
 * \fn new_temporal_stat_param()
 *
 * \return A new reference on allocated temporal_stat_param structure
 *
 * \brief Create a new temporal_stat_param reference 
 *
 */
Temporal_stat_param new_temporal_stat_param();

/**
 * \ingroup temporal_stat
 * \fn void free_temporal_stat_param(Temporal_stat_param p)
 *
 * \param p A reference on a temporal_stat_param structure
 *
 * \brief Free an allocated temporal_stat_param reference 
 *
 */
void free_temporal_stat_param(Temporal_stat_param p);

/**
 * \ingroup temporal_stat
 * \fn void define_temporal_stat_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/
void define_temporal_stat_param(char *extra_description);

/**
 * \ingroup temporal_stat
 * \fn Temporal_stat_param get_temporal_stat_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A temporal_stat_param that contains the parameters.
 */
Temporal_stat_param get_temporal_stat_param();

/**
 * \ingroup temporal_stat
 * \fn void temporal_stat(Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage);
 * \brief Compute the statistic of a list of images
 * 
 *
 * \param p is the structure of the parameters for operators
 * \param inputImages is a list of image to be used
 * \param outputImage is the image of result
 *
 */
void temporal_stat(Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage); 

/**
 * \ingroup temporal_stat
 * \fn void intensity_average(Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage);
 * \brief calculate the average intensity of a series
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImages is the images series
 * \param outputImage is the image of result
 *
 */
void intensity_average(Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage);

/**
 * \ingroup temporal_stat
 * \fn void sqrt_int_average(Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage)
 * \brief calculate the square root of the average intensity
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImages to use for the square root of the intensity average
 * \param outputImage is the image of result
 *
 */
void sqrt_int_average(Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage);

/**
 * \ingroup temporal_stat
 * \fn void amplitude_average (Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage)
 * \brief calculate the average amplitude of a series
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImages to use for the amplitude average
 * \param outputImage is the image of result
 *
 */
void amplitude_average (Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage);

/**
 * \ingroup temporal_stat
 * \fn void log_average (Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage)
 * \brief calculate the average logarithm of a series
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImages to use for the log average
 * \param outputImage is the image of result
 *
 */
void log_average (Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage);

/**
 * \ingroup temporal_stat
 * \fn void exp_log_average(Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage)
 * \brief calculate the exponential of logarithm
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImages to use for the standard deviation
 * \param outputImage is the image of result
 *
 */
void exp_log_average(Temporal_stat_param p, EFIDIRListImage inputImages, EFIDIRImage outputImage);

#endif
