/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

#ifndef COM_STATISTICS_H
#define COM_STATISTICS_H

#include "efidir_image.h"
#include "efidir_param.h"
#include "com_cal_slc.h"

/**
 * \ingroup statistic
 * \defgroup statistics Basic statistic operators
 * \author Flavien Vernier (LISTIC)
 */

/**
 * \ingroup statistics
 * \struct EFIDIRStat
 * 
 * \brief Statistics type
 *
 * EFIDIRStat stores statistics values
 *
 */
typedef struct EFIDIRStat{
  double min;                /**< minimum of values */
  int lineOfMin;             /**< line of the minimum*/
  int colOfMin;              /**< column of the minimum*/
  double max;                /**< maximum of values */
  int lineOfMax;             /**< line of the maximum */
  int colOfMax;              /**< column of the maximum */
  double sum;                /**< sum of values */
  double number_of_items;    /**< number of values */
  double average;            /**< average of values */
  double sum_sqr;             /**< sum of squared values */
  double sum_weight;      /**< sum of weight */
  double median;             /**< not used yet */
  double standard_deviation; /**< standard deviation of pixels values */
  double variance;           /**< variance */
  double RMS; /**< root mean square */
}EFIDIRstat, *EFIDIRStat;


/**
 * \ingroup statistics
 * \struct Statistics_param
 *
 * \brief Type that defines the parameters of the operator statistics
 *
 */
typedef struct Statistics_param{
  double threshold; /**< Any value lower than or equal to this threshold is not used to compute statisticsistic (default value -inf) */
}statistics_param, *Statistics_param;


/**
 * \ingroup statistics
 * \fn new_statistics()
 *
 * \return A new reference on allocated statistics structure
 *
 * \brief Create a new Statistics reference 
 *
 */
EFIDIRStat new_statistics();

/**
 * \ingroup statistics
 * \fn new_statistics_param()
 *
 * \return A new reference on allocated statistics_param structure
 *
 * \brief Create a new Statistics_param reference 
 *
 */
Statistics_param new_statistics_param();

/**
 * \ingroup statistics
 * \fn free_statistics_param(Statistics_param p)
 *
 * \param p A reference on a statistics_param structure
 *
 * \brief Free an allocated Statistics_param reference 
 *
 */
void free_statistics_param(Statistics_param p);

/**
 * \ingroup statistics
 * \fn free_statistics(EFIDIRStat p)
 *
 * \param p A reference on a EFIDIRStat structure
 *
 * \brief Free an allocated EFIDIRStat reference 
 *
 */
void free_statistics(EFIDIRStat p);

/**
 * \ingroup statistics
 * \fn void define_statistics_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/
void define_statistics_param(char *extra_description);

/**
 * \ingroup statistics
 * \fn Statistics_param get_statistics_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Statistics_param that contains the parameters.
 */
Statistics_param get_statistics_param();

/**
 * \ingroup statistics
 * \fn void statistics(Statistics_param p, EFIDIRImage inputImage, EFIDIRStat stats);
 * \brief Compute ordinary statistics of a part of image
 * The statistics operator computes ordinary statisticsistic values (MIN, MAX, AVERAGE...) of a given part of an image
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImage is the image to be processed
 * \param stats is the computed statisticsistic values
 *
 */
void statistics(Statistics_param p, EFIDIRImage inputImage, EFIDIRStat stats);


/**
 * \ingroup statistics
 * \fn void masked_weighted_statistics(Statistics_param p, EFIDIRImage inputImage, EFIDIRImage mask, EFIDIRImage weight, EFIDIRStat stats);
 * \brief Compute weighted statisticsistics of a part of image 
 * The statistics operator computes weighted statistics values (MIN, MAX, AVERAGE...) of a given part of an image
 *
 * \param p is the structure of the parameters for the operator
 * \param inputImage is the image to be processed
 * \param mask is the mask associated with the image to be processed
 * \param weight is the weight image associated with image to be processed
 * \param stats is the computed statisticsistic values
 *
 */
void masked_weighted_statistics(Statistics_param p, EFIDIRImage inputImage, EFIDIRImage mask, EFIDIRImage weight, EFIDIRStat stats);

/**
 * \ingroup statistics
 * \fn void compute_statistics(Statistics_param param_of_statistics, EFIDIRImage inputImage, EFIDIRStat stats);
 *
 * \brief Compute statisticsistics of an image
 *
 * The compute_statistics function computes, or reads from image header, the statisticsistic values (MIN, MAX, AVERAGE...) of a given image.
 * If not found, they will be computed and saved in the image header, providing that the image exists and its header is available.
 *
 * \param param_of_statistics is the structure of the parameters for the operator
 * \param inputImage is the image to be processed
 * \param stats is the computed statisticsistic values 
 *
 */
void compute_statistics(Statistics_param param_of_statistics, EFIDIRImage inputImage, EFIDIRStat stats);

/**
 * \ingroup statistics
 * \fn void save_statistics(Statistics_param param_of_statistics, EFIDIRImage inputImage, EFIDIRStat stats, char* file_mane);
 *
 * \brief Save statistics in a file
 *
 * Save the statistic values in a given file.
 *
 * /param param_of_statistics is the structure of the parameters for the operator
 * /param inputImage is the image to be processed
 * /param stats is the computed statisticsistic values
 * /param file_mane to save statistics
 */
void save_statistics(Statistics_param param_of_statistics, EFIDIRImage inputImage, EFIDIRStat stats, char* file_mane);
#endif /* STATISTICS_H */
